#include "matched_ids_proto_json_serializer.h"

#include <library/cpp/json/json_reader.h>
#include <library/cpp/json/json_writer.h>

#include <util/stream/str.h>

#include <algorithm>

using namespace NCrypta::NCm;
using namespace NJson;

namespace {
    const bool FORMAT_OUTPUT = false;
    const bool SORT_KEYS = false;
}

TString NMatchedIdsProtoJsonSerializer::Serialize(const TMatchedIdsProto& matchedIds) {
    TStringStream out;
    const auto& v = ToJsonValue(matchedIds);
    WriteJson(&out, &v, FORMAT_OUTPUT, SORT_KEYS);
    return out.Str();
}

NJson::TJsonValue NMatchedIdsProtoJsonSerializer::ToJsonValue(const TMatchedIdsProto& matchedIds) {
    TJsonValue ret(EJsonValueType::JSON_ARRAY);
    for (const auto& id : matchedIds) {
        TJsonValue jsonId(EJsonValueType::JSON_MAP);
        jsonId["type"] = id.GetId().GetType();
        jsonId["value"] = id.GetId().GetValue();
        jsonId["match_ts"] = id.GetMatchTs();
        jsonId["cas"] = id.GetCas();

        auto& attrs = jsonId["attributes"];
        attrs.SetType(EJsonValueType::JSON_MAP);
        if (id.AttributesSize() > 0) {
            for (const auto& attr : id.GetAttributes()) {
                attrs[attr.GetName()] = attr.GetValue();
            }
        }

        ret.AppendValue(jsonId);
    }
    return ret;
}

TMatchedIdsProto NMatchedIdsProtoJsonSerializer::Deserialize(const TString& json) {
    TMatchedIdsProto proto;
    Deserialize(json, proto);
    return proto;
}

void NMatchedIdsProtoJsonSerializer::Deserialize(const TString& json, TMatchedIdsProto& proto) {
    TJsonValue v;
    ReadJsonTree(json, &v);

    const auto& serializedIds = v.GetArray();
    proto.Reserve(serializedIds.size());

    for (const auto& id : serializedIds) {
        TMatchedIdProto matchedId;

        auto& id_ = *matchedId.MutableId();
        id_.SetType(id["type"].GetString());
        id_.SetValue(id["value"].GetString());

        matchedId.SetMatchTs(id["match_ts"].GetUInteger());
        matchedId.SetCas(id["cas"].GetUInteger());

        const auto& serializedAttrs = id["attributes"].GetMap();
        auto& attrs = *matchedId.MutableAttributes();
        attrs.Reserve(serializedAttrs.size());
        for (const auto& [key, value] : serializedAttrs) {
            auto& attribute = *attrs.Add();
            attribute.SetName(key);
            attribute.SetValue(value.GetString());
        }

        proto.Add()->Swap(&matchedId);
    }
}
