import argparse
import datetime
import logging
import os
import sys

import crypta.cm.services.db_stats.lib as db_stats
from crypta.lib.python.logging import logging_helpers
from crypta.lib.python import yaml_config
from crypta.lib.python.solomon import reporter
from crypta.lib.python.yql import client
from crypta.lib.python.yt import yt_helpers


logger = logging.getLogger(__name__)


INTERNAL_ID_TYPES = ["yandexuid", "icookie"]
SCHEDULING_TAG_FILTER_KEY = "scheduling-tag-filter"


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument("--config", required=True, type=yaml_config.load)
    return parser.parse_args()


def main():
    logging_helpers.configure_stdout_logger(logging.getLogger())

    args = parse_args()
    logger.info(args)
    config = args.config

    yt_pool = config["yt-pool"]
    db_path = config["db-path"]

    exit_code = 0
    for replica in config["replicas"]:
        try:
            yt_proxy = replica["yt-proxy"]
            dc = replica["dc"]
            logger.info("Get db statistics from %s", yt_proxy)

            yt_client = yt_helpers.get_yt_client(yt_proxy, yt_pool)
            operation_spec = {"scheduling_tag_filter": config[SCHEDULING_TAG_FILTER_KEY]} if SCHEDULING_TAG_FILTER_KEY in config else None
            yql_client = client.create_yql_client(
                yt_proxy,
                token=os.environ["YQL_TOKEN"],
                pool=yt_pool,
                tmp_folder=config["yt-tmp-dir"],
                operation_spec=operation_spec,
                max_job_count=config["max-job-count"],
                binary_cache_tmp_folder=config["yt-tmp-dir"],
                binary_cache_ttl=datetime.timedelta(seconds=config["binary-cache-ttl-sec"]),
            )
            solomon_client = reporter.create_solomon_reporter(common_labels={"dc": dc}, **config["solomon"])

            id_type_count = db_stats.get_id_type_count(yt_client, yql_client, db_path)
            logger.info("Id type counts:\n%s", id_type_count)

            touched_days_ago_count = db_stats.get_touched_days_ago(yt_client, yql_client, db_path, config["crypta-cm-udf-url"])
            logger.info("Touch by days ago:\n%s", touched_days_ago_count)

            ttl_by_type_count = db_stats.get_ttl_days_by_type(yt_client, yql_client, db_path, config["crypta-cm-udf-url"])
            logger.info("TTL days by type count:\n%s", ttl_by_type_count)

            if config["send-statistics-to-solomon"]:
                solomon_client.set_value("row.count", sum(id_type_count.values()))

                id_types, counts = zip(*id_type_count.items())
                solomon_client.set_value(["{}.{}.count".format("internal_id" if id_type in INTERNAL_ID_TYPES else "tag", id_type) for id_type in id_types], counts)

                touched_days_ago, counts = zip(*touched_days_ago_count.items())
                row_count = len(counts)
                solomon_client.set_value(["touch_count"] * row_count, counts, [{"days_ago": days_ago} for days_ago in touched_days_ago])

                row_count = len(ttl_by_type_count)
                counts = [int(row["count"]) for row in ttl_by_type_count]
                labels = [dict([(k, v) for k, v in row.items() if k != "count"]) for row in ttl_by_type_count]
                solomon_client.set_value(["ttl_count"] * row_count, counts, labels)

        except Exception:
            logger.exception("")
            exit_code = 1

    sys.exit(exit_code)
