import collections
import time

from crypta.cm.services.fpc_uploader.lib import schema
from crypta.lib.python import duid_upload
from crypta.lib.python.sampler import rest_sampler
from crypta.lib.python.yt.http_mapper import tvm_http_requests_mapper


URL_TEMPLATE_UPLOAD = "/upload?subclient=fpc-uploader"
BODY_TEMPLATE_UPLOAD = "{{\"ids\":[{{\"value\":\"{yandexuid}\",\"type\":\"yandexuid\"}}],\"ext_id\":{{\"value\":\"{value}\",\"type\":\"{type}\"}}}}"


class UploadFpcMapper(tvm_http_requests_mapper.TvmHttpRequestsMapper):
    def __init__(self, max_retries, sampler_percent, *args, **kwargs):
        super(UploadFpcMapper, self).__init__(*args, **kwargs)

        self.retry_counter = collections.Counter()
        self.max_retries = max_retries
        self.batch_size = 1
        self.sampler_percent = sampler_percent

    def get_request_wo_headers(self, batch):
        row = batch[0]

        yuid = row[schema.YUID]
        if yuid is None:  # TODO(levanov): Write to errors
            return

        if self.passes_sampling(yuid):
            for match in duid_upload.get_matches(row):
                yield "POST", self.url, BODY_TEMPLATE_UPLOAD.format(yandexuid=match.yandexuid, type=match.type, value=match.ext_id)

    def is_retriable(self, request):
        row = request.batch[0]
        key = duid_upload.get_row_key(row)

        return (self.retry_counter[key] < self.max_retries) and (request.status != "400")

    def retry_failed_request(self, request):
        row = request.batch[0]
        key = duid_upload.get_row_key(row)

        self.retry_counter.update([key])
        return self.is_retriable(request)

    def process_response(self, request):
        row = request.batch[0]
        key = duid_upload.get_row_key(row)

        if not request.ok and not self.is_retriable(request):
            yield {
                schema.YUID: row[schema.YUID],
                schema.DUID: row.get(schema.DUID),
                schema.STATUS: request.status,
                schema.DATA: request.response_body,
                schema.UNIXTIME: int(time.time()),
            }
        elif request.ok and key in self.retry_counter:
            del self.retry_counter[key]

    def passes_sampling(self, yuid):
        timestamp = yuid[-10:]
        return timestamp.isdigit() and rest_sampler.PassesLess(int(timestamp), denominator=100, rest=self.sampler_percent)
