import collections

import requests

from crypta.cm.services.identifier.lib import errors
from crypta.lib.python.yt.http_mapper import tvm_http_requests_mapper


URL_TEMPLATE_IDENTIFY = "/identify?subclient=identifier&type={tag}&value={ext_id}"


class IdentifyMapper(tvm_http_requests_mapper.TvmHttpRequestsMapper):
    def __init__(self, max_retries, *args, **kwargs):
        super(IdentifyMapper, self).__init__(*args, **kwargs)

        self.retry_counter = collections.Counter()
        self.max_retries = max_retries
        self.batch_size = 1

    def get_request_wo_headers(self, batch):
        yield "GET", self.url.format(**batch[0]), None

    def is_request_successful(self, request):
        return request.ok or request.status == str(requests.codes.not_found)

    def is_request_retriable(self, request):
        return self.retry_counter[self._get_counter_key(request.batch[0])] < self.max_retries

    def retry_failed_request(self, request):
        self.retry_counter.update([self._get_counter_key(request.batch[0])])
        return self.is_request_retriable(request)

    def process_response(self, request):
        row = request.batch[0]
        counter_key = self._get_counter_key(row)
        request_succeded = self.is_request_successful(request)

        if not request_succeded and not self.is_request_retriable(request):
            yield {
                errors.TAG: row["tag"],
                errors.EXT_ID: row["ext_id"],
                errors.STATUS: request.status,
                errors.DATA: request.response_body,
            }
        elif request_succeded and counter_key in self.retry_counter:
            del self.retry_counter[counter_key]

    def _get_counter_key(self, row):
        return (row["tag"], row["ext_id"])
