#include <crypta/cm/services/mutator/bin/cmd_args.pb.h>
#include <crypta/cm/services/mutator/lib/config/config.pb.h>
#include <crypta/cm/services/mutator/lib/parser.h>
#include <crypta/cm/services/mutator/lib/processor.h>
#include <crypta/lib/native/cmd_args/parse_pb_options.h>
#include <crypta/lib/native/log/log.h>
#include <crypta/lib/native/log/utils/log_exception.h>
#include <crypta/lib/native/pqlib/consumer.h>
#include <crypta/lib/native/pqlib/credentials_provider.h>
#include <crypta/lib/native/pqlib/logger.h>
#include <crypta/lib/native/pqlib/pqlib.h>
#include <crypta/lib/native/pqlib/reader.h>
#include <crypta/lib/native/signal_waiter/signal_waiter.h>
#include <crypta/lib/native/stats/graphite/graphite_stats_sink.h>
#include <crypta/lib/native/stats/log/log_stats_sink.h>
#include <crypta/lib/native/stats/solomon/solomon_stats_sink.h>
#include <crypta/lib/native/stats/stats.h>
#include <crypta/lib/native/stats/stats_writer.h>
#include <crypta/lib/native/time/shifted_clock.h>
#include <crypta/lib/native/yaml/parse_yaml_file.h>
#include <crypta/lib/native/yaml/yaml2proto.h>
#include <crypta/lib/native/yaml/ytree_node/utils.h>
#include <crypta/lib/native/yt/dyntables/async_database/kv_database.h>
#include <crypta/lib/native/yt/dyntables/async_database/logger.h>
#include <crypta/lib/native/yt/yson/utils.h>

#include <library/cpp/retry/retry.h>

#include <yt/yt/core/concurrency/thread_pool.h>
#include <yt/yt/core/misc/shutdown.h>

#include <util/generic/vector.h>
#include <util/stream/output.h>
#include <util/stream/str.h>

using namespace NCrypta;
using namespace NCrypta::NCm;
using namespace NCrypta::NCm::NMutator;
using namespace NCrypta::NLog;
using namespace NCrypta::NYtDynTables;

namespace {
    TConfig ParseArgs(int argc, const char** argv) {
        const auto& options = ParsePbOptions<TCmdArgs>(argc, argv);
        auto yamlConfig = ParseYamlFile(options.GetConfigFile());

        return Yaml2Proto<TConfig>(yamlConfig);
    }
}

int main(int argc, const char** argv) {
    TSignalWaiter stopSignalWaiter({SIGTERM, SIGINT});

    try {
        TShiftedClock::FreezeTimestampFromEnv();

        const auto& config = ParseArgs(argc, argv);

        RegisterLogs(config.GetLogs());

        auto log = NLog::GetLog("main");

        auto statsRegistry = Singleton<TStatsRegistry>();
        TStatsWriter statsWriter(*statsRegistry, TDuration::Minutes(1));
        if (config.HasGraphite()) {
            statsWriter.AddSink(MakeHolder<TGraphiteStatsSink>(config.GetGraphite()));
        }
        if (config.HasSolomon()) {
            statsWriter.AddSink(MakeHolder<TSolomonStatsSink>(config.GetSolomon()));
        }
        statsWriter.AddSink(MakeHolder<TLogStatsSink>("graphite"));
        statsWriter.Start();

        const auto& ytConfig = config.GetYt();
        NYtDynTables::NLogger::Configure(ytConfig.GetLogger());

        TStats::TSettings statsSettings;
        statsSettings.HistMax = 10;
        statsSettings.HistBinCount = 10;

        TStats kvStats("kv", statsSettings);
        TStats pqlibStats("pqlib", statsSettings);
        TStats processorStats("processor", statsSettings);

        const auto& logbrokerConfig = config.GetLogbroker();
        auto pqLibLogger = NPQ::NLogger::Create(logbrokerConfig.GetPqlibLogName());

        NPQ::TConsumer consumer(
                NPQ::NPQLib::Create(logbrokerConfig.GetPqlib()),
                pqLibLogger,
                logbrokerConfig.GetConsumer(),
                NPQ::NCredentialsProvider::Create(logbrokerConfig.GetCredentials(), pqLibLogger),
                pqlibStats
        );

        auto asyncDatabase = MakeAtomicShared<TKvAsyncDatabase>(ytConfig.GetClient(), ytConfig.GetTable(), ytConfig.GetModifyRows());
        TKvDatabase database(asyncDatabase, kvStats);

        THashSet<TString> trackedBackRefTags(config.GetTrackedBackRefTags().begin(), config.GetTrackedBackRefTags().end());

        const auto& processorConfig = config.GetProcessor();
        TVector<THandlerQueue> handlerQueues(processorConfig.GetCount());

        auto processorThreadPool = NYT::New<NYT::NConcurrency::TThreadPool>(processorConfig.GetThreads(), "Processor");

        for (auto& handlerQueue : handlerQueues) {
            processorThreadPool->GetInvoker()->Invoke(BIND(&TProcessor::Run, NYT::New<TProcessor>(
                    handlerQueue,
                    database,
                    processorConfig.GetMaxCommandsPerTransaction(),
                    MakeRetryOptions(processorConfig.GetCommitRetryOptions()),
                    TDuration::MilliSeconds(processorConfig.GetMaxBatchingTimeMs()),
                    trackedBackRefTags,
                    processorConfig.GetDeduplicationCache().GetSize() / processorConfig.GetCount(),
                    TDuration::Seconds(processorConfig.GetDeduplicationCache().GetMaxAgeSec()),
                    processorStats
            )));
        }

        const auto& evacuateLogProducerConfig = logbrokerConfig.GetEvacuateLogProducer();
        NPQ::TProducer evacuateLogProducer(
                NPQ::NPQLib::Create(logbrokerConfig.GetPqlib()),
                evacuateLogProducerConfig,
                NPQ::NCredentialsProvider::Create(logbrokerConfig.GetCredentials(), NPQ::NLogger::Create(evacuateLogProducerConfig.GetLogName())),
                statsSettings
        );

        THandlerFactory handlerFactory(trackedBackRefTags, config.GetTtl(), evacuateLogProducer, processorStats);

        NPQ::TCookieQueue cookiesToCommit;
        TParser parser(handlerFactory, cookiesToCommit, handlerQueues, processorStats);

        auto parserThreadPool = NYT::New<NYT::NConcurrency::TThreadPool>(config.GetParser().GetThreads(), "Parser");

        NPQ::TReader reader(
            TDuration::MilliSeconds(config.GetReader().GetReadTimeoutMs()),
            consumer,
            cookiesToCommit,
            [&parser, &parserThreadPool] (auto&& readResult) {
                parserThreadPool->GetInvoker()->Invoke(BIND([&parser, readResult = std::move(readResult)] () mutable {
                    parser.Parse(std::move(readResult));
                }));
            },
            NLog::GetLog(logbrokerConfig.GetPqlibLogName())
        );
        reader.SetOnFinishHandler([&stopSignalWaiter] () {
            stopSignalWaiter.Signal();
        });

        log->info("Start");
        reader.Start();

        stopSignalWaiter.Wait();

        log->info("Stop");
    } catch (const yexception& e) {
        LogException(e.what());
        return 1;
    } catch (const std::exception& e) {
        LogException(e.what());
        return 1;
    } catch (...) {
        NLog::LogException("Unknown exception");
        return 1;
    }

    Cerr << "Shutting down YT library" << Endl;
    NYT::Shutdown();

    Cerr << "Exiting" << Endl;
    return 0;
}
