#include "touch_handler.h"

#include "ttl_selector.h"
#include "utils.h"

#include <crypta/cm/services/common/serializers/id/string/id_string_serializer.h>

using namespace NCrypta::NCm;
using namespace NCrypta::NCm::NMutator;

TTouchHandler::TTouchHandler(
    TInstant commandTimestamp,
    TTouchCommand touchCommand,
    const TTtlConfig& ttlConfig,
    TStats& stats)
    : THandler(commandTimestamp, stats, "touch")
    , TouchCommand(std::move(touchCommand))
    , TtlSelector(ttlConfig)
{
}

TString TTouchHandler::GetShardingKey() const {
    return TouchCommand.ShardingKey;
}

TMaybe<TString> TTouchHandler::GetDeduplicationKey() const {
    return "T" + NIdSerializer::ToString(TouchCommand.ExtId);
}

TVector<TId> TTouchHandler::GetLookupIds() {
    return { TouchCommand.ExtId };
}

void TTouchHandler::UpdateDbState(TDbState& dbState) {
    const auto& extId = TouchCommand.ExtId;
    const auto* dbMatch = dbState.GetMatches().Get(extId);

    if (dbMatch) {
        const auto& oldTouch = dbMatch->GetTouch();
        const auto& newTouch = TouchCommand.TouchTs;

        if (const auto& ttl = SelectTtl(oldTouch, newTouch, extId.Type); ttl.Defined()) {
            Log->info("TTL: {}, touch: {} -> {}", ttl->Seconds(), oldTouch.Seconds(), newTouch.Seconds());
            dbState.TouchMatch(extId, newTouch, *ttl);

            Stats.Count->Add(GetTtlDaysMetric(*ttl, "touch"));
        } else {
            Log->info("No ttl: old {}, new {}", oldTouch.Seconds(), newTouch.Seconds());
        }
    }
}

TMaybe<TDuration> TTouchHandler::SelectTtl(TInstant oldTouch, TInstant newTouch, const TString& extIdType) const {
    TMaybe<TDuration> result;
    if (TtlSelector.IsExtendTtlTimeoutElapsed(oldTouch, newTouch)) {
        Log->info("Selected extended ttl");
        result = TtlSelector.GetExtendedTtl(extIdType);
    } else if (TtlSelector.IsTouchTimeoutElapsed(oldTouch, newTouch)) {
        Log->info("Selected default ttl");
        result = TtlSelector.GetDefaultTtl(extIdType);
    }

    return result;
}

void TTouchHandler::OnCommitSuccess() {
    Stats.Count->Add("touch.total");
}
