#include "ttl_selector.h"

#include <crypta/cm/services/common/expiration/need_touch.h>

#include <util/generic/maybe.h>

using namespace NCrypta::NCm::NMutator;

namespace {
    TMaybe<ui64> GetCustomTagTtl(const TString& tag, const TTtlConfig& ttlConfig) {
        TMaybe<ui64> result;

        const auto& customTagTtls = ttlConfig.GetCustomTagTtls();
        auto it = customTagTtls.find(tag);
        if (it != customTagTtls.end()) {
            result = it->second;
        }

        return result;
    }
}

TTtlSelector::TTtlSelector(const TTtlConfig& ttlConfig)
    : TtlConfig(ttlConfig)
{
}

TDuration TTtlSelector::GetDefaultTtl(const TString& extIdType) const {
    const auto& customTtl = GetCustomTagTtl(extIdType, TtlConfig);
    return TDuration::Seconds(customTtl.Defined() ? customTtl.GetRef() : TtlConfig.GetDefaultTtl());
}

TDuration TTtlSelector::GetExtendedTtl(const TString& extIdType) const {
    const auto& customTtl = GetCustomTagTtl(extIdType, TtlConfig);
    return TDuration::Seconds(customTtl.Defined() ? customTtl.GetRef() : TtlConfig.GetExtendedTtl());
}

bool TTtlSelector::IsTouchTimeoutElapsed(TInstant oldTouchTime, TInstant newTouchTime) const {
    return NeedTouch(oldTouchTime, newTouchTime, TDuration::Seconds(TtlConfig.GetTouchTimeoutSec()));
}

bool TTtlSelector::IsExtendTtlTimeoutElapsed(TInstant oldTouchTime, TInstant newTouchTime) const {
    return NeedTouch(oldTouchTime, newTouchTime, TDuration::Seconds(TtlConfig.GetExtendTtlTimeoutSec()));
}

bool TTtlSelector::SetAppropriateTtl(TMatch& incomingMatch, const TMatch* dbMatch) const {
    const auto& extIdType = incomingMatch.GetExtId().Type;

    bool ttlIsChanged = false;

    if (dbMatch == nullptr){
        incomingMatch.SetTtl(GetDefaultTtl(extIdType));
        ttlIsChanged = true;
    } else {
        if (IsExtendTtlTimeoutElapsed(dbMatch->GetTouch(), incomingMatch.GetTouch())) {
            incomingMatch.SetTtl(GetExtendedTtl(extIdType));
        } else {
            incomingMatch.SetTtl(GetDefaultTtl(extIdType));
        }
        // Even if touch timeout is not elapsed, the default ttl can be changed from config:
        ttlIsChanged = dbMatch->GetTtl() != incomingMatch.GetTtl();
    }

    return ttlIsChanged;
}
