#include "upload_handler.h"

#include "ttl_selector.h"
#include "utils.h"

#include <crypta/cm/services/common/data/id_utils.h>
#include <crypta/cm/services/common/data/match_comparator.h>
#include <crypta/cm/services/common/data/proto_helpers.h>
#include <crypta/cm/services/common/proto/refugee.pb.h>
#include <crypta/lib/native/time/shifted_clock.h>
#include <library/cpp/protobuf/json/proto2json.h>

using namespace NCrypta;
using namespace NCrypta::NCm;
using namespace NCrypta::NCm::NMutator;

namespace {
    bool AreIdsChanged(const TMatch& incomingMatch, const TMatch& dbMatch) {
        for (const auto& [type, incomingMatchedId]: incomingMatch.GetInternalIds()) {
            const auto& it = dbMatch.GetInternalIds().find(type);
            if (it == dbMatch.GetInternalIds().end()) {
                return true;
            }

            const auto& currentMatchedId = it->second;

            if (incomingMatchedId.GetMatchTs() > currentMatchedId.GetMatchTs() &&
                !NMatchedIdComparator::Equal(incomingMatchedId, currentMatchedId, NMatchedIdComparator::EMode::Id | NMatchedIdComparator::EMode::Attributes)
            ) {
                return true;
            }
        }
        return false;
    }

    bool AreBackReferencesTracked(const TMatch& incomingMatch, const THashSet<TString>& trackedBackRefTags) {
        return incomingMatch.GetTrackBackReference() || trackedBackRefTags.contains(incomingMatch.GetExtId().Type);
    }
}

TUploadHandler::TUploadHandler(
        TInstant commandTimestamp,
        TUploadCommand uploadCommand,
        const THashSet<TString>& trackedBackRefTags,
        const TTtlConfig& ttlConfig,
        NPQ::IProducer& evacuateLogProducer,
        TStats& stats)
    : THandler(commandTimestamp, stats, "upload")
    , UploadCommand(std::move(uploadCommand))
    , TrackedBackRefTags(trackedBackRefTags)
    , TtlSelector(ttlConfig)
    , EvacuateLogProducer(evacuateLogProducer)
{
}

TString TUploadHandler::GetShardingKey() const {
    return UploadCommand.ShardingKey;
}

TVector<NCm::TId> TUploadHandler::GetLookupIds() {
    TVector<TId> ids;

    ids.emplace_back(UploadCommand.IncomingMatch.GetExtId());

    if (AreBackReferencesTracked(UploadCommand.IncomingMatch, TrackedBackRefTags)) {
        for (const auto&[type, internalId]: UploadCommand.IncomingMatch.GetInternalIds()) {
            ids.push_back(internalId.GetId());
        }
    }

    return ids;
}

void TUploadHandler::UpdateDbState(TDbState& dbState) {
    const auto* dbMatch = dbState.GetMatches().Get(UploadCommand.IncomingMatch.GetExtId());

    auto matchToWrite = UploadCommand.IncomingMatch;
    bool ttlIsChanged = TtlSelector.SetAppropriateTtl(matchToWrite, dbMatch);

    if (dbMatch == nullptr || ttlIsChanged || TtlSelector.IsTouchTimeoutElapsed(dbMatch->GetTouch(), matchToWrite.GetTouch()) || AreIdsChanged(matchToWrite, *dbMatch)) {
        dbState.WriteMatch(matchToWrite);
        Stats.Count->Add(GetTtlDaysMetric(matchToWrite.GetTtl(), "upload"));

        WriteEvacuateLog();
    }
}

void TUploadHandler::OnCommitSuccess() {
    Stats.Count->Add("upload.total");
}

void TUploadHandler::WriteEvacuateLog() {
    if (UploadCommand.IncomingMatch.GetExtId().Type != DUID_TYPE) {
        return;
    }

    crypta::cm::proto::TRefugee refugee;
    refugee.SetTimestamp(TInstant::Now().Seconds());
    refugee.SetType("cm");

    auto extIdProto = MakeIdProto(UploadCommand.IncomingMatch.GetExtId());
    refugee.MutableSource()->Swap(&extIdProto);

    for (const auto& [type, internalId] : UploadCommand.IncomingMatch.GetInternalIds()) {
        if (type == YANDEXUID_TYPE) {
            auto internalIdProto = MakeIdProto(internalId.GetId());
            refugee.MutableDestination()->Swap(&internalIdProto);

            EvacuateLogProducer.TryEnqueue(NProtobufJson::Proto2Json(refugee));
        }
    }
}
