#include "parser.h"

#include <crypta/cm/services/common/data/id_hash.h>

#include <yt/yt/core/concurrency/scheduler.h>

#include <util/string/split.h>

using namespace NCrypta::NCm;
using namespace NCrypta::NCm::NMutator;

namespace {
    const THash<TString> HASH;
    const char* const SALT = "15969763710089819020";
}

TParser::TParser(THandlerFactory& handlerFactory, NPQ::TCookieQueue& cookiesToCommit, TVector<THandlerQueue>& handlerQueues, TStats& stats)
    : HandlerFactory(handlerFactory)
    , CookiesToCommit(cookiesToCommit)
    , HandlerQueues(handlerQueues)
    , Log(NLog::GetLog("parser"))
    , Stats(stats)
{
}

void TParser::Parse(NCrypta::NPQ::TConsumer::TReadResult&& readResult) {
    TVector<NYT::TFuture<void>> commitFutures;
    TVector<THandlers> shardedHandlers(HandlerQueues.size());

    for (const auto& item : readResult.Data) {
        for (const auto& line : StringSplitter(item).Split('\n').SkipEmpty()) {
            try {
                auto handler = HandlerFactory.CreateHandler(line);

                commitFutures.push_back(handler->GetCommitFuture());

                size_t shard = HASH(handler->GetShardingKey() + SALT) % shardedHandlers.size();
                shardedHandlers[shard].push_back(std::move(handler));
            } catch (const yexception& e) {
                Stats.Count->Add("parser.errors.bad_command");
                Log->error("Bad command: {}. Error: {}", line.Token(), e.what());
            }
        }
    }

    readResult.ClearData();

    for (size_t i = 0; i < HandlerQueues.size(); i++) {
        if (shardedHandlers[i].empty()) {
            continue;
        }

        HandlerQueues[i].Enqueue(std::move(shardedHandlers[i]));
    }

    Stats.Percentile->Add("parser.read.commands", commitFutures.size());
    Stats.Count->Add("parser.read.total.commands", commitFutures.size());

    NYT::NConcurrency::WaitFor(NYT::AllSucceeded(commitFutures)).ThrowOnError();

    CookiesToCommit.Enqueue(readResult.EpochCookie);
}
