#include "processor.h"

#include <crypta/cm/services/common/db_state/db_state_loader.h>
#include <crypta/cm/services/common/db_state/get_changes_batch.h>
#include <crypta/lib/native/time/scope_timer.h>
#include <crypta/lib/native/time/sleeper.h>
#include <crypta/lib/native/yt/dyntables/async_database/yt_exception.h>

#include <yt/yt/core/concurrency/delayed_executor.h>
#include <util/stream/str.h>

using namespace NCrypta;
using namespace NCrypta::NCm::NMutator;

bool TProcessor::TBatch::operator>(const TBatch& other) const {
    return NextTry > other.NextTry;
}

TProcessor::TProcessor(
        THandlerQueue& handlerQueue,
        NYtDynTables::TKvDatabase& database,
        size_t maxCommandsPerTransaction,
        TRetryOptions commitRetryOptions,
        TDuration maxBatchingTime,
        const THashSet<TString>& trackedBackRefTags,
        ui64 deduplicationCacheSize,
        TDuration cacheMaxAge,
        TStats& stats
)
    : HandlerQueue(handlerQueue)
    , CacheMaxAge(cacheMaxAge)
    , HandlerCache(deduplicationCacheSize)
    , Database(database)
    , MaxCommandsPerTransaction(maxCommandsPerTransaction)
    , MaxBatchingTime(maxBatchingTime)
    , CommitRetryOptions(std::move(commitRetryOptions))
    , TrackedBackRefTags(trackedBackRefTags)
    , Log(NLog::GetLog("processor"))
    , Stats(stats)
{
}

void TProcessor::Run() {
    while (true) {
        auto batch = GetBatch();

        if (batch.Handlers.empty()) {
            continue;
        }

        TScopeTimer timer(Stats.Percentile, "timing.process_per_batch");
        Stats.Percentile->Add("batch.count.command", batch.Handlers.size());

        const auto& ids = GetLookupIds(batch);

        if (!ApplyChanges(ids, batch)) {
            Stats.Count->Add("batch.retry");
            batch.NextTry = CommitRetryOptions.GetTimeToSleep(batch.Retries).ToDeadLine();
            ++batch.Retries;
            Batches.push_back(std::move(batch));
            PushHeap(Batches.begin(), Batches.end(), std::greater<TBatch>());
        } else {
            CommitSuccess(batch);
        }
    }
}

TProcessor::TBatch TProcessor::GetBatch() {
    const auto& now = TInstant::Now();

    if (!Batches.empty() && (now >= Batches.front().NextTry)) {
        PopHeap(Batches.begin(), Batches.end(), std::greater<TBatch>());
        auto batch = std::move(Batches.back());
        Batches.pop_back();
        return batch;
    }

    if (DequeuedHandlers.empty()) {
        NYT::NConcurrency::TDelayedExecutor::WaitForDuration(LastDequeueTime + MaxBatchingTime - TInstant::Now());
    }

    if (DequeuedHandlers.size() < MaxCommandsPerTransaction) {
        TVector<THandlers> handlerVectors;
        HandlerQueue.DequeueAll(&handlerVectors);
        LastDequeueTime = TInstant::Now();

        const auto& cacheDeadline = now + CacheMaxAge;

        for (auto& handlerVector : handlerVectors) {
            for (auto& handler : handlerVector) {
                const auto& deduplicationKey = handler->GetDeduplicationKey();
                if (deduplicationKey.Defined()) {
                    auto it = HandlerCache.Find(deduplicationKey.GetRef());
                    if (it != HandlerCache.End() && it.Value() < cacheDeadline) {
                        handler->CommitSuccess();
                        Stats.Count->Add("command.skipped");
                        continue;
                    }
                    HandlerCache.Update(deduplicationKey.GetRef(), now);
                }
                DequeuedHandlers.emplace_back(std::move(handler));
            }
        }
    }

    auto batchEnd = (DequeuedHandlers.size() > MaxCommandsPerTransaction) ? (DequeuedHandlers.begin() + MaxCommandsPerTransaction) : DequeuedHandlers.end();

    THandlers handlers(std::make_move_iterator(DequeuedHandlers.begin()), std::make_move_iterator(batchEnd));
    DequeuedHandlers.erase(DequeuedHandlers.begin(), batchEnd);

    return TBatch{
        .Handlers = std::move(handlers)
    };
}

TProcessor::TIds TProcessor::GetLookupIds(TBatch& batch) {
    TScopeTimer timer(Stats.Percentile, "timing.get_lookup_ids");

    TIds lookupIds;

    for (auto& handler : batch.Handlers) {
        const auto& ids = handler->GetLookupIds();
        lookupIds.insert(lookupIds.end(), ids.begin(), ids.end());
    }

    return lookupIds;
}

bool TProcessor::ApplyChanges(const TIds& lookupIds, TBatch& batch) {
    TScopeTimer timer(Stats.Percentile, "timing.apply_changes");
    try {
        auto tx = Database.StartTransaction();
        TDbStateLoader stateLoader(Log, TrackedBackRefTags);
        auto dbState = stateLoader.Load(*tx, lookupIds, true);

        for (auto& handler : batch.Handlers) {
            handler->UpdateDbState(dbState);
        }

        const auto& changesBatch = GetChangesBatch(dbState);
        batch.WriteCount = changesBatch.RecordsToUpdate.size();
        batch.DeleteCount = changesBatch.KeysToDelete.size();

        if (batch.WriteCount > 0 || batch.DeleteCount > 0) {
            tx->Write(changesBatch.RecordsToUpdate);
            tx->Delete(changesBatch.KeysToDelete);

            tx->Commit();
        }
    } catch (const NCrypta::NYtDynTables::TYtException& e) {
        Log->warn(e.what());
        return false;
    }

    return true;
}

void TProcessor::CommitSuccess(TBatch& batch) {
    TScopeTimer timer(Stats.Percentile, "timing.on_commit_success");

    for (const auto& handler : batch.Handlers) {
        handler->CommitSuccess();
    }

    Stats.Hist->Add("retries", batch.Retries);
    Stats.Count->Add("batch.success");
    Stats.Percentile->Add("batch.count.write", batch.WriteCount);
    Stats.Percentile->Add("batch.count.delete", batch.DeleteCount);
}

TInstant TProcessor::THandlerWeighter::Weight(const TInstant& commandTimestamp) {
    return commandTimestamp;
}
