#include "quoter_server.h"

#include "quoter_service_impl.h"

#include <crypta/lib/native/grpc/keepalive/keepalive.h>
#include <crypta/lib/native/grpc/logging/logging_interceptor_factory.h>
#include <crypta/lib/native/grpc/stats/stats_interceptor_factory.h>
#include <crypta/lib/native/singleton/tagged_singleton.h>

#include <util/string/cast.h>

using namespace NCrypta::NCm::NQuoter;

TQuoterServer::TQuoterServer(ui32 port, const TQuoterState& quoterState, const TStats::TSettings& statsSettings)
    : Port(port)
    , Service(MakeHolder<TQuoterServiceImpl>(quoterState, statsSettings))
    , Log(NLog::GetLog("quoter_server"))
    , StatsSettings(statsSettings)
{
}

void TQuoterServer::Start() {
    const auto& hostPort = TString("0.0.0.0:") + ToString(Port);

    grpc::ServerBuilder builder;
    builder.AddListeningPort(hostPort, grpc::InsecureServerCredentials());
    builder.RegisterService(Service.Get());
    builder.SetMaxReceiveMessageSize(1024 * 1024);

    NCrypta::NGrpc::EnableServerKeepalive(builder);

    std::vector<std::unique_ptr<grpc::experimental::ServerInterceptorFactoryInterface>> creators;

    creators.push_back(std::make_unique<NCrypta::NGrpc::TLoggingInterceptorFactory>(
        [&](TString&& line){ Log->info(std::move(line)); },
        Log
    ));
    creators.push_back(std::make_unique<NCrypta::NGrpc::TStatsInterceptorFactory>(StatsSettings));

    builder.experimental().SetInterceptorCreators(std::move(creators));

    Server = builder.BuildAndStart();

    if (!Server) {
        ythrow yexception() << "Can't start server";
    }

    Log->info("TQuoterServer is listening on {}", hostPort);
}

TQuoterServer::~TQuoterServer() {
    Log->info("Shutting down the server...");
    Server->Shutdown();
    Server->Wait();
    Log->info("Done");
}
