#include "quoter_state_updater.h"

#include <crypta/lib/native/graphite/utils.h>
#include <crypta/lib/native/singleton/tagged_singleton.h>
#include <crypta/lib/native/solomon/solomon_exception.h>
#include <crypta/lib/native/time/scope_timer.h>

#include <util/generic/vector.h>
#include <util/string/builder.h>

#include <algorithm>

using namespace NCrypta::NCm::NQuoter;

const TString SOLOMON_PROJECT_YT = "yt";

namespace {
    template<typename TStringMap>
    TString MapToStableString(const TStringMap& map) {
        TVector<TString> keys;
        for (const auto& it: map) {
            keys.push_back(it.first);
        }

        std::sort(keys.begin(), keys.end());

        TStringBuilder result;
        bool first = true;
        for (const auto& key: keys) {
            if (first) {
                first = false;
            } else {
                result << " ";
            }

            result << key << "=" << map.at(key);
        }

        return result;
    }
}

TQuoterStateUpdater::TQuoterStateUpdater(
    TQuoterState& quoterState,
    const TDuration interval,
    const TSolomonConfig& solomonConfig,
    const TDuration solomonRequestTimeout,
    const TDyntablesPolicy& dyntablesPolicy,
    const TStats::TSettings& statsSettings)
    : TPeriodicTask(interval, [this](){ UpdateState(); }, "StateUpdater", true)
    , QuoterState(quoterState)
    , SolomonDataClient(
        solomonConfig.GetSchema(),
        solomonConfig.GetHost(),
        solomonConfig.GetPort(),
        solomonConfig.GetOauthToken(),
        solomonRequestTimeout,
        SOLOMON_PROJECT_YT
    )
    , DyntablesPolicy(dyntablesPolicy)
    , Log(NLog::GetLog("state_updater"))
    , Stats(TaggedSingleton<TStats, decltype(*this)>("state_updater", statsSettings))
{
    TQuotaState emptyState;
    emptyState.SetIsFull(false);
    emptyState.SetDescription("Haven't received actual state yet");

    const auto& envTypes = DyntablesPolicy.GetEnvironmentTypes();
    for (auto it = envTypes.begin(); it != envTypes.end(); ++it) {
        QuoterState.Set(it->first, emptyState);
    }

    Start();
}

void TQuoterStateUpdater::UpdateState() {
    for (const auto& [envType, clusterSettings]: DyntablesPolicy.GetEnvironmentTypes()) {
        TQuotaState newState;

        TStringBuilder description;
        description << envType << ":\n";

        for (size_t i = 0; i < clusterSettings.ClustersSize(); ++i) {
            const auto& baseLabels = clusterSettings.GetClusters()[i].GetLabels();

            TSolomonException::TCode code = HTTP_OK;
            try {
                auto current = GetSolomonMetric(baseLabels, DyntablesPolicy.GetSensorCurrent());
                auto limit = GetSolomonMetric(baseLabels, DyntablesPolicy.GetSensorLimit());

                auto minGbLeft = clusterSettings.GetMinGbLeft();
                auto gbLeft = limit - current;

                if (gbLeft < minGbLeft) {
                    newState.SetIsFull(true);
                }

                description << MapToStableString(baseLabels)  << ": "
                            << (newState.GetIsFull() ? "Not enough quota, " : "Enough quota, ")
                            << gbLeft << " GB left ("
                            << current << " GB used out of "
                            << limit << " GB, min_gb_left = " << minGbLeft << ")" << "\n";
            } catch (const TSolomonException& e) {
                code = e.Code;
                Log->error("Exception while making query to Solomon: {}", e.what());
                Stats.Count->Add(MakeGraphiteMetric("state", envType, "errors"));
                description << MapToStableString(baseLabels) << ": Exception while making query to Solomon: " << e.what() << "\n";
            }

            Stats.Count->Add("solomon.request.http_code." + ToString(code));
        }

        newState.SetDescription(description);

        Log->debug("New state: is_full={}, description={}", newState.GetIsFull(), newState.GetDescription());
        QuoterState.Set(envType, newState);
        Stats.Count->Add(MakeGraphiteMetric("state", envType, "status", (newState.GetIsFull() ? "full" : "not_full")));
    }
}

double TQuoterStateUpdater::GetSolomonMetric(const TProtoStringMap& baseLabels, const TString& sensor) const {
    TScopeTimer scopeTimer(Stats.Percentile, "solomon.latency");
    auto allLabels = baseLabels;
    allLabels["sensor"] = sensor;

    return SolomonDataClient.GetScalarSensor(
        allLabels,
        DyntablesPolicy.GetAggregationFunction(),
        TDuration::Seconds(DyntablesPolicy.GetAggregationWindowSec()));
}
