#include "parser.h"

#include <crypta/cm/services/common/data/id_hash.h>
#include <crypta/cm/services/rt_duid_uploader/common/lib/experiment_checker/experiment_checker.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/resource/resource.h>

#include <util/digest/murmur.h>
#include <util/stream/zlib.h>
#include <util/string/split.h>
#include <util/string/cast.h>

using namespace NCrypta::NCm;
using namespace NCrypta::NCm::NRtDuidUploader;

namespace {
    constexpr const char * const YUID = "yuid";
    constexpr const char * const DOMAIN_COOKIE = "domain_cookie";
    constexpr const char * const TIMESTAMP = "timestamp";
    constexpr const char * const SOURCE = "source";

    TMatch ParseRow(const TStringBuf& row) {
        NJson::TJsonValue rowJson;
        NJson::ReadJsonTree(row, &rowJson, true);

        TMatch result;
        result.Yuid = rowJson[YUID].GetStringSafe();
        result.Duid = rowJson[DOMAIN_COOKIE].GetStringSafe();
        result.Timestamp = TInstant::Seconds(rowJson[TIMESTAMP].GetIntegerSafe());
        result.Source = rowJson[SOURCE].GetStringSafe();
        return result;
    }

    TParser::TBannedYuids GetBannedYuids(TString fileName="banned.txt.gz") {
        TParser::TBannedYuids result;
        TFileInput fileInput(fileName);
        TBufferedZLibDecompress inputStream{&fileInput};

        TString s;
        while (inputStream.ReadLine(s) > 0) {
            if (!s.empty()) {
                result.insert(FromStringWithDefault<ui64>(s, 0));
            }
        }

        return result;
    }
}

TParser::TParser(const TParserConfig& config, TUploader& uploader, NPQ::TCookieQueue& cookiesToCommit, TStats& stats)
    : ThreadPool(NYT::New<NYT::NConcurrency::TThreadPool>(config.GetThreads(), "Parser"))
    , Sampler(100, config.GetSamplePercent(), TRestSampler::EMode::Less)
    , Uploader(uploader)
    , CookiesToCommit(cookiesToCommit)
    , BannedSources(config.GetBannedSources().begin(), config.GetBannedSources().end())
    , ExperimentSources(config.GetExperimentSources().begin(), config.GetExperimentSources().end())
    , Log(NLog::GetLog("parser"))
    , Stats(stats)
    , BannedYuids(GetBannedYuids())
{
}

void TParser::ScheduleParse(NPQ::TConsumer::TReadResult&& readResult) {
    ThreadPool->GetInvoker()->Invoke(
        BIND(&TParser::Parse, this, NYT::Passed(std::move(readResult)))
    );
}

void TParser::Parse(NCrypta::NPQ::TConsumer::TReadResult&& readResult) {
    TVector<NYT::TFuture<void>> commitFutures;

    Log->debug("Got {} messages, cookie={}", readResult.Data.size(), readResult.EpochCookie.Cookie);

    for (const auto& item : readResult.Data) {
        for (const auto& it : StringSplitter(item).Split('\n').SkipEmpty()) {
            try {
                auto match = ParseRow(it.Token());

                ui64 yuid64 = FromStringWithDefault<ui64>(match.Yuid, 0);
                if (BannedYuids.contains(yuid64)) {
                    Stats.Count->Add("dropped_by_ban." + match.Source);
                    continue;
                }

                if (ExperimentSources.contains(match.Source) && !NExperimentChecker::ShouldUploadYandexuid(match.Yuid)) {
                    Stats.Count->Add("dropped_by_experiment." + match.Source);
                    continue;
                }

                if (BannedSources.contains(match.Source)) {
                    Stats.Count->Add("banned." + match.Source);
                    continue;
                }

                if (Sampler.Passes(match.Duid)) {
                    Stats.Count->Add("uploaded." + match.Source);
                    commitFutures.push_back(Uploader.Upload(std::move(match)));
                }
            } catch (const yexception& e) {
                Stats.Count->Add("errors.bad_command");
                Log->error("Bad command: {}. Error: {}", it.Token(), e.what());
            }
        }
    }

    readResult.ClearData();

    Stats.Percentile->Add("read.commands", commitFutures.size());

    Log->debug("Waiting for uploads, cookie={}", readResult.EpochCookie.Cookie);

    NYT::NConcurrency::WaitFor(NYT::AllSucceeded(commitFutures)).ThrowOnError();

    Log->debug("Uploads complete, cookie={}", readResult.EpochCookie.Cookie);

    CookiesToCommit.Enqueue(readResult.EpochCookie);
}
