#pragma once

#include "request_factory.h"

#include <crypta/cm/services/rt_duid_uploader/common/lib/config/cmd_args.pb.h>
#include <crypta/cm/services/rt_duid_uploader/common/lib/config/config.pb.h>
#include <crypta/cm/services/rt_duid_uploader/common/lib/parser.h>
#include <crypta/cm/services/rt_duid_uploader/common/lib/uploader.h>

#include <crypta/lib/native/cmd_args/parse_pb_options.h>
#include <crypta/lib/native/log/log.h>
#include <crypta/lib/native/log/utils/log_exception.h>
#include <crypta/lib/native/pqlib/consumer.h>
#include <crypta/lib/native/pqlib/credentials_provider.h>
#include <crypta/lib/native/pqlib/logger.h>
#include <crypta/lib/native/pqlib/pqlib.h>
#include <crypta/lib/native/pqlib/reader.h>
#include <crypta/lib/native/signal_waiter/signal_waiter.h>
#include <crypta/lib/native/stats/log/log_stats_sink.h>
#include <crypta/lib/native/stats/solomon/solomon_stats_sink.h>
#include <crypta/lib/native/stats/stats.h>
#include <crypta/lib/native/stats/stats_writer.h>
#include <crypta/lib/native/time/shifted_clock.h>
#include <crypta/lib/native/yaml/parse_yaml_file.h>
#include <crypta/lib/native/yaml/yaml2proto.h>

#include <yt/yt/core/concurrency/thread_pool.h>
#include <yt/yt/core/misc/shutdown.h>

#include <util/generic/vector.h>
#include <util/stream/output.h>
#include <util/stream/str.h>

namespace NCrypta::NCm::NRtDuidUploader {
    template <typename TFactory, typename TFactoryConfig>
    int Run(int argc, const char** argv) {
        TSignalWaiter stopSignalWaiter({SIGTERM, SIGINT});

        try {
            const auto& options = ParsePbOptions<TCmdArgs>(argc, argv);

            auto yamlConfig = ParseYamlFile(options.GetConfigFile());
            const auto& config = Yaml2Proto<TConfig>(yamlConfig);

            RegisterLogs(config.GetLogs());

            auto factoryYamlConfig = ParseYamlFile(options.GetFactoryConfigFile());
            const TFactory factory(Yaml2Proto<TFactoryConfig>(factoryYamlConfig));

            auto statsRegistry = Singleton<TStatsRegistry>();
            TStatsWriter statsWriter(*statsRegistry, TDuration::Minutes(1));
            if (config.HasSolomon()) {
                statsWriter.AddSink(MakeHolder<TSolomonStatsSink>(config.GetSolomon()));
            }
            statsWriter.AddSink(MakeHolder<TLogStatsSink>("graphite"));
            statsWriter.Start();

            TStats::TSettings statsSettings;
            statsSettings.HistMax = 10;
            statsSettings.HistBinCount = 10;

            TStats pqlibStats("pqlib", statsSettings);
            TStats pasrerStats("parser", statsSettings);
            TStats uploaderStats("uploader", statsSettings);

            const auto& logbrokerConfig = config.GetLogbroker();
            auto pqLibLogger = NPQ::NLogger::Create(logbrokerConfig.GetPqlibLogName());

            NPQ::TConsumer consumer(
                NPQ::NPQLib::Create(logbrokerConfig.GetPqlib()),
                pqLibLogger,
                logbrokerConfig.GetConsumer(),
                NPQ::NCredentialsProvider::Create(logbrokerConfig.GetCredentials(), pqLibLogger),
                pqlibStats
            );

            NPQ::TCookieQueue cookiesToCommit;

            auto uploader = NYT::New<TUploader>(config.GetUploader(), factory, uploaderStats);
            TParser parser(config.GetParser(), *uploader, cookiesToCommit, pasrerStats);

            NPQ::TReader reader(
                TDuration::MilliSeconds(config.GetReader().GetReadTimeoutMs()),
                consumer,
                cookiesToCommit,
                [&parser] (auto&& readResult) mutable {
                    parser.ScheduleParse(std::move(readResult));
                },
                NLog::GetLog(logbrokerConfig.GetPqlibLogName())
            );
            reader.SetOnFinishHandler([&stopSignalWaiter] () {
                stopSignalWaiter.Signal();
            });

            NLog::GetLog("main")->info("Start");
            reader.Start();

            stopSignalWaiter.Wait();

            NLog::GetLog("main")->info("Stop");
        } catch (const yexception& e) {
            NLog::LogException(e.what());
            return 1;
        } catch (const std::exception& e) {
            NLog::LogException(e.what());
            return 1;
        } catch (...) {
            NLog::LogException("Unknown exception");
            return 1;
        }

        Cerr << "Shutting down YT library" << Endl;
        NYT::Shutdown();

        Cerr << "Exiting" << Endl;
        return 0;
    }
}
