#include "uploader.h"

#include <crypta/lib/native/time/scope_timer.h>
#include <crypta/lib/native/time/sleeper.h>
#include <crypta/lib/native/yt/dyntables/async_database/yt_exception.h>

#include <yt/yt/core/concurrency/config.h>
#include <yt/yt/core/concurrency/delayed_executor.h>

#include <util/system/guard.h>
#include <util/stream/str.h>

using namespace NCrypta;
using namespace NCrypta::NCm;
using namespace NCrypta::NCm::NRtDuidUploader;

TUploader::TUploader(
    const TUploaderConfig& config,
    const IRequestFactory& requestFactory,
    TStats& stats
)
    : InFlightSemaphore(config.GetMaxInFlightRequests())
    , Throttler(NYT::NConcurrency::CreateReconfigurableThroughputThrottler(NYT::New<NYT::NConcurrency::TThroughputThrottlerConfig>(config.GetMaxRps())))
    , RequestTimeout(TDuration::MilliSeconds(config.GetRequestTimeoutMs()))
    , RetryTimeout(TDuration::MilliSeconds(config.GetRetryTimeoutMs()))
    , ThreadPool(NYT::New<NYT::NConcurrency::TThreadPool>(config.GetThreads(), "Uploader"))
    , RequestFactory(requestFactory)
    , Client("HttpClient")
    , Log(NLog::GetLog("uploader"))
    , Stats(stats)
{
}

NYT::TFuture<void> TUploader::Upload(TMatch match) {
    Log->debug("Scheduling yuid={}, duid={}", match.Yuid, match.Duid);

    return Throttler->Throttle(1).Apply(
        BIND(
            &TUploader::AsyncUpload,
            NYT::MakeStrong(this),
            NYT::Passed(std::move(match)),
            NYT::Passed(TGuard(InFlightSemaphore))
        ).Via(ThreadPool->GetInvoker())
    );
}

void TUploader::AsyncUpload(TMatch match, TGuard<TFastSemaphore> guard) {
    Y_UNUSED(guard);
    const auto& request = RequestFactory.CreateRequest(match);
    size_t retries = 0;

    while(!TryUpload(request)) {
        NYT::NConcurrency::TDelayedExecutor::WaitForDuration(RetryTimeout);
        ++retries;
    }

    Stats.Percentile->Add("upload.latency", (TInstant::Now() - match.Timestamp).Seconds());
    Stats.Hist->Add("upload.retries", retries);
    Stats.Count->Add("upload.success");
}

bool TUploader::TryUpload(const NNeh::TMessage& message) {
    const auto& response = Client.Request(message, RequestTimeout);
    if (!response.IsOK()) {
        Log->warn("Failed to upload: {}", response.GetMessage());
        return false;
    } else if (response.Value()->IsError()) {
        Log->warn("Failed to upload: {}", response.Value()->GetErrorText());
        return false;
    }
    return true;
}
