import datetime
import logging
import sys

import yt.wrapper as yt

from crypta.cm.services.toucher.lib import (
    errors,
    touch_mapper,
)
from crypta.lib.python.tvm import helpers as tvm
from crypta.lib.python.yt import yt_helpers
from crypta.lib.python.yt.http_mapper import utils as yt_http_utils


logger = logging.getLogger(__name__)


def send_table_to_cm(yt_client, src_table, err_table, hosts, job_config, tvm_secret):
    err_table = yt.TablePath(err_table, schema=errors.get_schema())
    yt_client.create("table", err_table, recursive=True, ignore_existing=True)

    with yt_client.Transaction():
        logger.info("Sending table %s to CM", src_table)
        logger.info("Writing errors to %s", err_table)

        extra_spec = {"secure_vault": {
            touch_mapper.TouchMapper.TVM_SECRET_ENV_VAR: tvm_secret,
        }}
        tvm_test_port = tvm.get_tvm_test_port()
        if tvm_test_port:
            extra_spec["secure_vault"][tvm.TVM_TEST_PORT_ENV_VAR] = tvm_test_port

        yt_http_utils.run_map(touch_mapper.TouchMapper, hosts, job_config, src_table, err_table, additional_spec=extra_spec, yt_client=yt_client)
        logger.info("Done.")

        logger.info("Dropping table %s", src_table)
        yt_client.remove(src_table)


def run(config, logger):
    yt_client = yt_helpers.get_yt_client(config.Yt.Proxy, config.Yt.Pool)

    tables = yt_client.list(config.InputDir)
    if not tables:
        logger.info("No tables found. Exiting.")
        return
    logger.info("Found the following tables: %s", tables)

    job_config = yt_http_utils.job_config_from_proto(config.JobConfig)

    exit_code = 0
    for table in tables:
        try:
            src_table = yt.ypath_join(config.InputDir, table)
            err_table = yt.ypath_join(config.ErrorsDir, table)
            backup_table = yt.ypath_join(config.BackupDir, table)
            ttl = datetime.timedelta(days=config.BackupTtlDays)

            yt_helpers.make_backup_table(src_table, backup_table, ttl, yt_client=yt_client)

            send_table_to_cm(
                yt_client,
                src_table,
                err_table,
                config.DstHosts,
                job_config,
                config.TvmSecret,
            )
            yt_helpers.set_ttl(err_table, ttl, remove_if_empty=True, yt_client=yt_client)

        except Exception:
            logger.exception("Failed when sending %s to CM", table)
            exit_code = 1

    sys.exit(exit_code)
