import collections
import time

from library.python.protobuf.json import proto2json

from crypta.cm.services.api.lib.logic.touch.request.proto import touch_request_body_pb2
from crypta.cm.services.toucher.lib import errors
from crypta.lib.proto.identifiers import id_pb2
from crypta.lib.python.yt.http_mapper import tvm_http_requests_mapper


TYPE = "type"
VALUE = "value"
TOUCH_TS = "touch_ts"


def get_body(batch):
    proto_ids = touch_request_body_pb2.TTouchRequestBody(Items=[
        touch_request_body_pb2.TTouchRequestBody.TItem(
            Id=id_pb2.TId(Type=row[TYPE], Value=row[VALUE]),
            TouchTimestamp=row[TOUCH_TS],
        )
        for row in batch
    ])
    return proto2json.proto2json(proto_ids, proto2json.Proto2JsonConfig(field_name_mode=proto2json.FldNameMode.FieldNameSnakeCase))


class TouchMapper(tvm_http_requests_mapper.TvmHttpRequestsMapper):
    def __init__(self, max_retries, *args, **kwargs):
        super(TouchMapper, self).__init__(*args, **kwargs)

        self.retry_counter = collections.Counter()
        self.max_retries = max_retries

    def get_request_wo_headers(self, batch):
        yield "POST", self.url, get_body(batch)

    def is_request_retriable(self, request):
        return self.retry_counter[self._get_counter_key(request)] < self.max_retries

    def retry_failed_request(self, request):
        self.retry_counter.update([self._get_counter_key(request)])
        return self.is_request_retriable(request)

    def process_response(self, request):
        counter_key = self._get_counter_key(request)
        request_succeded = request.ok or request.status == "404"

        if not request_succeded and not self.is_request_retriable(request):
            for row in request.batch:
                yield {
                    errors.BATCH_ID: ":".join(counter_key),
                    errors.TYPE: row[TYPE],
                    errors.VALUE: row[VALUE],
                    errors.STATUS: request.status,
                    errors.DATA: request.response_body,
                    errors.UNIXTIME: int(time.time()),
                }
        elif request_succeded and counter_key in self.retry_counter:
            del self.retry_counter[counter_key]

    def _get_counter_key(self, request):
        return request.batch[0][TYPE], request.batch[0][VALUE]
