import argparse
import datetime
import logging
import os
import sys

import yt.wrapper as yt

from crypta.cm.services.uploader.lib import (
    config_fields,
    errors,
    upload_match_mapper
)
from crypta.lib.python.logging import logging_helpers
from crypta.lib.python import yaml_config
from crypta.lib.python.yt import yt_helpers
from crypta.lib.python.yt.http_mapper import utils as yt_http_utils


TVM_SECRET_ENV_VAR = "TVM_SECRET"

logger = logging.getLogger(__name__)


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument("--config", required=True, type=yaml_config.load)
    return parser.parse_args()


def get_job_config(config):
    job_config = yt_http_utils.tvm_job_config_from_args(config[config_fields.JOB_CONFIG], [config_fields.MAX_RETRIES, config_fields.FROZEN_TIME])
    job_config.url_template = upload_match_mapper.URL_TEMPLATE_UPLOAD
    return job_config


def send_table_to_cm(src_table, err_table, hosts, job_config, tvm_secret, drop_src):
    err_table = yt.TablePath(err_table, schema=errors.get_schema())
    yt.create("table", err_table, recursive=True, ignore_existing=True)

    with yt.Transaction():
        logger.info("Sending table %s to CM", src_table)
        logger.info("Writing errors to %s", err_table)

        secure_vault_spec = {"secure_vault": {upload_match_mapper.UploadMatchMapper.TVM_SECRET_ENV_VAR: tvm_secret}}
        yt_http_utils.run_map(upload_match_mapper.UploadMatchMapper, hosts, job_config, src_table, err_table, additional_spec=secure_vault_spec)
        logger.info("Done.")

        if drop_src:
            logger.info("Dropping table %s", src_table)
            yt.remove(src_table)
        else:
            logger.info("Not dropping tables.")


def main():
    logging_helpers.configure_stdout_logger(logging.getLogger())
    logger.info("================ Start ================")

    args = parse_args()
    config = args.config
    logger.info(config)

    yt.config.set_proxy(config[config_fields.YT_PROXY])
    yt.config["pool"] = config[config_fields.YT_POOL]

    src_dir = config[config_fields.SOURCE_DIR]

    tables = yt.list(src_dir)
    if not tables:
        logger.info("No tables found. Exiting.")
        sys.exit(0)
    logger.info("Found the following tables: %s", tables)

    job_config = get_job_config(config)

    exit_code = 0
    for table in tables:
        try:
            src_table = yt.ypath_join(src_dir, table)
            err_table = yt.ypath_join(config[config_fields.ERRORS_DIR], table)
            backup_table = yt.ypath_join(config[config_fields.BACKUP_DIR], table)

            yt_helpers.make_backup_table(src_table, backup_table, datetime.timedelta(days=config[config_fields.BACKUP_TTL_DAYS]))

            tvm_secret = os.environ[TVM_SECRET_ENV_VAR]
            send_table_to_cm(src_table, err_table, config[config_fields.DESTINATION_HOSTS], job_config, tvm_secret, config[config_fields.DROP_SRC])
            yt_helpers.set_ttl(err_table, datetime.timedelta(days=config[config_fields.BACKUP_TTL_DAYS]), remove_if_empty=True)

        except Exception:
            logger.exception("Failed when sending %s to CM", table)
            exit_code = 1

    logger.info("================ Finish ================")
    sys.exit(exit_code)
