# encoding: utf-8

from datetime import timedelta as dt

import yt.wrapper as yt

import crypta.cm.services.db_stats.lib as db_stats_lib
from crypta.lib.python import crypta_env
from crypta.lib.python.data_size import DataSize
from crypta.lib.python.solomon.proto import alert_pb2
from crypta.lib.python.spine import (
    arcadia_ci,
    logfeller,
)
from crypta.lib.python.spine.consts import environment
from crypta.lib.python.spine.consts.yt_proxy import YtProxy
from crypta.lib.python.spine.juggler import (
    consts,
    juggler_check_generator,
)
from crypta.lib.python.spine.juggler.flap_detector_params import FlapDetectorParams
from crypta.lib.python.spine.sandbox import sandbox_scheduler
from crypta.lib.python.spine.solomon import (
    solomon_alert_registry,
    solomon_alert_utils,
    solomon_check_generator,
)
from crypta.lib.python.spine.yasm import yasm_dashboard_generator
from crypta.lib.python.spine.yt import (
    yt_config_registry,
    yt_replicated_table,
)
from crypta.lib.python.spine.yt.yt_latency_metric import YtLatencyMetric
from crypta.lib.python.spine.yt.yt_size_metric import YtSizeMetric
from sandbox.projects.crypta import run_universal_bundle
from sandbox.projects.crypta.cm import (
    fpc_uploader,
    identifier,
)

CRYPTA_CM_TEST = "cm-test.crypta.yandex.net"
CRYPTA_CM_PROD = "cm.crypta.yandex.net"
CRYPTA_CM_INTERNATIONAL_PROD = "cm-intl.crypta.yandex.net"
PROJECT = "crypta_cm"

DYNTABLE_MAP_REDUCE_YT_DEPENDENCIES = {
    environment.STABLE: [YtProxy.seneca_vla],
    environment.TESTING: [YtProxy.zeno],
}
INTERNATIONAL_DYNTABLE_MAP_REDUCE_YT_DEPENDENCIES = {
    environment.STABLE: [YtProxy.vapnik],
}

DYNTABLE_PROXY_STABLE_GROUP = YtProxy.Group.dyntable_prod_without_man
ALL_DYNTABLE_YT_DEPENDENCIES = {
    environment.STABLE: DYNTABLE_PROXY_STABLE_GROUP,
    environment.TESTING: [YtProxy.zeno],
}

OFFLINE_PROXY_GROUP = YtProxy.Group.offline
OFFLINE_YT_DEPENDENCIES = {
    environment.STABLE: OFFLINE_PROXY_GROUP,
    environment.TESTING: OFFLINE_PROXY_GROUP,
}

SENECA_VLA_PLUS_HAHN = [YtProxy.seneca_vla] + list(OFFLINE_PROXY_GROUP)
OFFLINE_AND_DYNTABLE_MAP_REDUCE_YT_DEPENDENCIES = {
    environment.STABLE: SENECA_VLA_PLUS_HAHN,
    environment.TESTING: SENECA_VLA_PLUS_HAHN,
}

CLUSTER_WITH_DYNAMIC_STORE_READ = YtProxy.seneca_vla


def add_yt_replicated_table(juggler):
    root_path = "//home/crypta/production/cookie_matching/rt/db"
    master = yt_replicated_table.Master(
        name="markov",
        proxy=YtProxy.markov,
        path=yt.ypath_join(root_path, "master"),
        expected_attributes={
            "tablet_cell_bundle": "crypta-cm",
            "replicated_table_options": {
                "tablet_cell_bundle_name_ttl": 300000,
                "tablet_cell_bundle_name_failure_interval": 60000,
                "max_sync_replica_count": 1,
                "enable_replicated_table_tracker": True,
                "sync_replica_lag_threshold": 600000,
            },
        },
        replication_lag_threshold=dt(seconds=60),
        sync_count=1,
    )

    replicas = [
        yt_replicated_table.Replica(
            name=proxy.split(".", 1)[0],
            proxy=proxy,
            path=path,
            expected_attributes=expected_attributes,
            expected_replication_attributes=expected_replication_attributes,
        )
        for proxy in DYNTABLE_PROXY_STABLE_GROUP
        for path, expected_attributes, expected_replication_attributes in [
            (yt.ypath_join(root_path, "replica"), {
                "tablet_cell_bundle": "crypta-cm",
                "in_memory_mode": "uncompressed",
                "enable_dynamic_store_read": proxy == CLUSTER_WITH_DYNAMIC_STORE_READ,
            }, {
                "replicated_table_tracker_enabled": True,
            }),
            ("//home/bigb/crypta_replicas/replica", {}, {
                "replicated_table_tracker_enabled": False,
            })
        ]
    ]

    return yt_replicated_table.ReplicatedTableConfigRegistry(juggler, yt_replicated_table.ReplicatedTable(master, replicas))


def get_registry():
    juggler = juggler_check_generator.CryptaYtCheckGenerator(tags=["crypta-cm"])
    yt_juggler_gen = juggler.clone(host=CRYPTA_CM_PROD)

    yt_registry = yt_config_registry.CryptaYtConfigRegistry(yt_juggler_gen)

    yasm = juggler.add_subregistry(yasm_dashboard_generator.CryptaYtYasmDashboardGenerator("Cookie Matching"))

    release_registry = arcadia_ci.CryptaReleaseGenerator(yt_juggler_gen).standard_release(
        title="Cookie Matching",
        abs_paths=["crypta/cm/**"],
    )

    errors_paths = [
        ("cookie_matching/rt/db_sync/errors/prepare_to_sync_db", DataSize(mb=1), OFFLINE_PROXY_GROUP),

        ("cookie_matching/rt/fpc_upload/errors", DataSize(mb=1), OFFLINE_PROXY_GROUP),
        ("cookie_matching/rt/db_sync/errors/uploader", DataSize(mb=1), OFFLINE_PROXY_GROUP),
        ("cookie_matching/rt/db_sync/errors/identifier", DataSize(mb=50), OFFLINE_PROXY_GROUP),
        ("tmp/errors/get-rows-to-expire", DataSize(mb=0), DYNTABLE_PROXY_STABLE_GROUP),
        ("cookie_matching/rt/deleter/errors", DataSize(mb=1), OFFLINE_PROXY_GROUP),
        ("cookie_matching/offline/expirator/errors", DataSize(mb=1), DYNTABLE_PROXY_STABLE_GROUP),
    ]

    for path, threshold, proxies in errors_paths:
        for proxy in proxies:
            check = YtSizeMetric(yt_registry, path, yt_proxy=proxy).add_disk_space_alert(
                predicate=alert_pb2.GT,
                threshold=threshold,
            )
            if proxies == DYNTABLE_PROXY_STABLE_GROUP:
                check.add_nodata_mode(consts.NoDataMode.force_ok)

    for proxy in DYNTABLE_PROXY_STABLE_GROUP:
        YtSizeMetric(yt_registry, "cookie_matching/rt/db/replica", yt_proxy=proxy).add_unmerged_row_counts_alert(
            predicate=alert_pb2.LT,
            threshold=10**9,
        )

    add_yt_replicated_table(yt_juggler_gen)

    for proxy in OFFLINE_PROXY_GROUP:
        for path in ("cookie_matching/rt/fpc_upload/to_upload", "cookie_matching/rt/db_sync/to_identify"):
            YtSizeMetric(yt_registry, path, yt_proxy=proxy)
            YtLatencyMetric(yt_registry, path, yt_proxy=proxy)

    for stream in [
        "crypta@prod@cm-access-log",
        "crypta@test@cm-access-log",
        "crypta@prod@cm-change-log",
        "crypta@test@cm-change-log",
        "crypta@prod@evacuation-rt-log",
        "crypta@test@evacuation-rt-log",
    ]:
        logfeller.add_unprased_records_check(juggler, stream)

    add_sandbox(juggler)
    add_api_checks(juggler, yasm)
    add_mutator_checks(juggler, yasm)
    add_quoter(juggler, release_registry)
    add_rt_duid_uploader_checks(juggler, yasm)
    add_db_stats(juggler)

    return juggler


def add_sandbox(juggler_check_generator):
    def get_uploader_tvm_secret_env(env):
        return crypta_env.to_sandbox_secrets_env(crypta_env.VARS_BY_ENV[env].cm_uploader_tvm)

    expirator_extra_params = {
        "handle": "expire",
        "subclient": "expirator",
    }

    russian_installation = {
        "name": "russia",
        "host_env": [
            (CRYPTA_CM_PROD, environment.STABLE),
            (CRYPTA_CM_TEST, environment.TESTING),
        ],
        "task_params": [
            (fpc_uploader.CryptaCmFpcUploaderTask, dt(hours=6), OFFLINE_YT_DEPENDENCIES, dt(minutes=10), dt(hours=6)),
            (identifier.CryptaCmIdentifierTask, dt(hours=24), OFFLINE_YT_DEPENDENCIES, dt(hours=1), dt(hours=3)),
        ],
        "universal_task_params": [
            ("crypta-cm-calc-expire", dt(hours=24), dt(hours=3), dt(hours=26), DYNTABLE_MAP_REDUCE_YT_DEPENDENCIES, None),
            ("crypta-cm-db-backup", dt(hours=24), dt(hours=3), dt(hours=26), OFFLINE_AND_DYNTABLE_MAP_REDUCE_YT_DEPENDENCIES, None),
            ("crypta-cm-db-stats", dt(hours=3), dt(hours=10), dt(hours=24), DYNTABLE_MAP_REDUCE_YT_DEPENDENCIES, None),
            ("crypta-cm-ext-id-mapper", dt(minutes=30), dt(hours=14), dt(hours=24), DYNTABLE_MAP_REDUCE_YT_DEPENDENCIES, get_uploader_tvm_secret_env),
            ("crypta-cm-prepare-touch", dt(hours=4), dt(hours=4), dt(days=2), OFFLINE_YT_DEPENDENCIES, None),
            ("crypta-cm-toucher", dt(hours=1), dt(hours=24), dt(hours=12), OFFLINE_YT_DEPENDENCIES, get_uploader_tvm_secret_env),
            ("crypta-cm-uploader", dt(minutes=30), dt(hours=3), dt(hours=24), OFFLINE_YT_DEPENDENCIES, get_uploader_tvm_secret_env),
        ],
    }

    international_installation = {
        "name": "international",
        "host_env": [
            (CRYPTA_CM_INTERNATIONAL_PROD, environment.STABLE),
        ],
        "task_params": [
        ],
        "universal_task_params": [
            ("crypta-cm-calc-expire", dt(hours=24), dt(hours=3), dt(hours=26), INTERNATIONAL_DYNTABLE_MAP_REDUCE_YT_DEPENDENCIES, None),
            ("crypta-cm-db-stats", dt(hours=3), dt(hours=10), dt(hours=24), INTERNATIONAL_DYNTABLE_MAP_REDUCE_YT_DEPENDENCIES, None),
            ("crypta-cm-ext-id-mapper", dt(minutes=30), dt(hours=14), dt(hours=24), INTERNATIONAL_DYNTABLE_MAP_REDUCE_YT_DEPENDENCIES, get_uploader_tvm_secret_env),
            ("crypta-cm-prepare-touch", dt(hours=4), dt(hours=4), dt(days=2), OFFLINE_YT_DEPENDENCIES, None),
            ("crypta-cm-toucher", dt(hours=1), dt(hours=24), dt(hours=12), OFFLINE_YT_DEPENDENCIES, get_uploader_tvm_secret_env),
            ("crypta-cm-uploader", dt(minutes=30), dt(hours=3), dt(hours=24), OFFLINE_YT_DEPENDENCIES, get_uploader_tvm_secret_env),
        ],
    }

    for installation in [russian_installation, international_installation]:
        installation_name = installation["name"]

        for host, env in installation["host_env"]:
            sandbox = sandbox_scheduler.create_default_generator(juggler_check_generator.clone(host=host), ["CM"])

            for task, crit_time, yt_deps, schedule_interval, kill_timeout in installation["task_params"]:
                extra_params = {"installation": installation_name}

                sandbox.create_scheduler(
                    task,
                    schedule_interval=schedule_interval,
                    kill_timeout=kill_timeout,
                    env=env,
                    extra_params=extra_params,
                ).check(
                    crit_time,
                ).add_yt_dependencies(
                    yt_deps[env],
                )

            for bundle_name, schedule_interval, kill_timeout, crit_time, yt_deps, secrets_env_getter in installation["universal_task_params"]:
                juggler_service = None
                template_vars = {
                    "installation": installation_name
                }

                if bundle_name == "crypta-cm-ext-id-mapper":
                    juggler_service = run_universal_bundle.CryptaRunUniversalBundle._join_not_empty(bundle_name, "expire", installation_name)
                    template_vars.update(expirator_extra_params)

                sandbox.create_run_universal_bundle_scheduler(
                    bundle_name=bundle_name,
                    env=env,
                    cmd=["{{cwd}}/" + bundle_name, "--config", "{{cwd}}/config.yaml"],
                    schedule_interval=schedule_interval,
                    kill_timeout=kill_timeout,
                    sequential_run=True,
                    secrets_env=secrets_env_getter(env) if secrets_env_getter else None,
                    juggler_service=juggler_service,
                    semaphore_suffix=installation_name,
                    template_rendering_context=template_vars,
                ).check(
                    crit_time=crit_time
                ).add_yt_dependencies(
                    yt_deps[env],
                )

        deleter_handle = "delete"
        deleter_template_vars = {
            "installation": installation_name,
            "handle": deleter_handle,
            "subclient": "deleter",
        }

        deleter_bundle_name = "crypta-cm-ext-id-mapper"
        deleter_env = environment.STABLE

        sandbox.create_run_universal_bundle_scheduler(
            bundle_name=deleter_bundle_name,
            env=deleter_env,
            cmd=["{{cwd}}/" + deleter_bundle_name, "--config", "{{cwd}}/config.yaml"],
            secrets_env=get_uploader_tvm_secret_env(deleter_env),
            juggler_service=run_universal_bundle.CryptaRunUniversalBundle._join_not_empty(deleter_bundle_name, deleter_handle, installation_name),
            semaphore_suffix=installation_name,
            template_rendering_context=deleter_template_vars,
            start_immediately=False,
        )


def add_api_checks(juggler_check_generator, yasm):
    yasm.add_devops(service="API", prj_tag="crypta-cm-api")

    for dc, env, nanny_service in [
        ("sas", environment.TESTING, "test_crypta_cm_api_sas"),
        ("sas", environment.PRODUCTION, "prod_crypta_cm_api_sas"),
        ("vla", environment.PRODUCTION, "prod_crypta_cm_api_vla"),
    ]:
        flap_detector_params = FlapDetectorParams(dt(minutes=5), dt(minutes=10))

        juggler = juggler_check_generator.clone(
            child_group=nanny_service,
            child_group_type=consts.GroupType.nanny,
            escalation=(env == environment.PRODUCTION),
            warn_limit=0,
            crit_limit="10%",
        )

        juggler.icmpping().add_flap_detector(flap_detector_params)
        juggler.http(service="alive", path="/ping?subclient=monitoring", port=80).add_flap_detector(flap_detector_params)
        juggler.tvm_client_status()

        solomon = solomon_check_generator.DcSolomonCheckGenerator(
            juggler,
            project=PROJECT,
            service="api",
            cluster=env,
            dc=dc,
        )

        if env == environment.PRODUCTION:
            solomon.get_sensor("identify.latency.wall_time.p95").create_threshold_check(
                aggregation=alert_pb2.MAX,
                predicate=alert_pb2.GT,
                threshold=20000,
                period=dt(minutes=2),
                description="latency (μs): {{ pointValue }}",
            ).add_flap_detector(flap_detector_params)

            solomon.get_sensor("identify.request.total.replied").create_threshold_check(
                aggregation=alert_pb2.AVG,
                predicate=alert_pb2.LT,
                threshold=100,
                period=dt(minutes=2),
                description="rpm: {{ pointValue }}",
            ).add_flap_detector(flap_detector_params)

            solomon.get_sensor("upload.latency.wall_time.p95").create_threshold_check(
                aggregation=alert_pb2.MAX,
                predicate=alert_pb2.GT,
                threshold=10000,
                period=dt(minutes=2),
                description="latency (μs): {{ pointValue }}",
            ).add_flap_detector(flap_detector_params)

            solomon.create_sensor_aggregation_check_by_host(
                service="5XX_codes",
                period=dt(minutes=2),
                sensor_glob="*.request.http_code.5??",
                predicate=alert_pb2.GT,
                threshold=600,
                description="Total 5XX codes : {{expression.total}} > {{expression.threshold}}",
            ).add_flap_detector(flap_detector_params).add_nodata_mode(consts.NoDataMode.force_ok)

            solomon.get_sensor("quoter_client.quota_state.full").create_threshold_check(
                aggregation=alert_pb2.AVG,
                predicate=alert_pb2.GT,
                threshold=1,
                period=dt(minutes=2),
                description="Quota is FULL as reported by: {{ pointValue }} nodes",
            ).add_flap_detector(flap_detector_params).add_nodata_mode(consts.NoDataMode.force_ok)


def add_mutator_checks(juggler_check_generator, yasm):
    yasm.add_devops(service="Mutator", prj_tag="crypta-cm-mutator")

    service = "mutator"

    for dc, env, nanny_service in [
        ("sas", environment.TESTING, "test_crypta_cm_mutator_sas"),
        ("sas", environment.PRODUCTION, "prod_crypta_cm_mutator_sas"),
        ("vla", environment.PRODUCTION, "prod_crypta_cm_mutator_vla"),
    ]:
        flap_detector_params = FlapDetectorParams(dt(minutes=5), dt(minutes=10))

        juggler = juggler_check_generator.clone(
            child_group=nanny_service,
            child_group_type=consts.GroupType.nanny,
            escalation=(env == environment.PRODUCTION),
            warn_limit=0,
            crit_limit=1,
        )
        juggler.icmpping().add_flap_detector(flap_detector_params)

        solomon = solomon_check_generator.DcSolomonCheckGenerator(
            juggler,
            project=PROJECT,
            service=service,
            cluster=env,
            dc=dc,
        )

        for sensor in [
            "processor.errors.invalid_command_format",
            "processor.errors.unknown_command",
        ]:
            solomon.get_sensor(sensor).create_threshold_check(
                aggregation=alert_pb2.MAX,
                predicate=alert_pb2.GT,
                threshold=0,
                period=dt(minutes=5),
                description="parsing errors (msgs): {{ pointValue }}",
            ).add_nodata_mode(consts.NoDataMode.force_ok)

    solomon = solomon_check_generator.ServiceSolomonCheckGenerator(
        juggler_check_generator.clone(
            escalation=False,
        ),
        project=PROJECT,
        service=service,
        cluster=environment.PRODUCTION,
    )
    solomon.create_sensor_aggregation_check(
        service="total_commands_processed",
        period=dt(minutes=5),
        sensor_glob="processor.*.total",
        predicate=alert_pb2.LT,
        threshold=5000,
        description="Total commands processed : {{expression.total}} < {{expression.threshold}}",
    )
    solomon.create_sensor_aggregation_check(
        service="expire_commands_processed",
        period=dt(days=1),
        sensor_glob="processor.expire.total",
        predicate=alert_pb2.LT,
        threshold=10**6,
        description="Expire commands processed : {{expression.total}} < {{expression.threshold}}",
        time_aggregation="sum",
    )

    solomon.create_logbroker_lag_check(
        consumer="crypta/prod/cm/consumer",
        topic="crypta/prod/cm/change-log",
        threshold=100 * 1000,
        period=dt(minutes=5),
    ).add_flap_detector(FlapDetectorParams(
        dt(minutes=15),
        dt(minutes=30),
    )).add_phone_escalation()


def add_quoter(juggler_check_generator, release_registry):
    release_registry.add_nanny_bundle(
        "CRYPTA_CM_QUOTER_BUNDLE",
        "crypta/cm/services/quoter/bundle/crypta-cm-quoter.json",
    )

    for env, nanny_service in [
        (environment.TESTING, "test_crypta_cm_quoter"),
        (environment.PRODUCTION, "prod_crypta_cm_quoter"),
    ]:
        flap_detector_params = FlapDetectorParams(dt(minutes=5), dt(minutes=10))

        juggler = juggler_check_generator.clone(
            child_group=nanny_service,
            child_group_type=consts.GroupType.nanny,
            escalation=False,
            warn_limit=0,
            crit_limit="50%",
        )

        juggler.icmpping().add_flap_detector(flap_detector_params)

        solomon = solomon_check_generator.ServiceSolomonCheckGenerator(
            juggler,
            project=PROJECT,
            service="quoter",
            cluster=env,
        )

        if env == environment.PRODUCTION:
            solomon.get_sensor("quoter_server.timing.get_quota_state.p95").create_threshold_check(
                aggregation=alert_pb2.MAX,
                predicate=alert_pb2.GT,
                threshold=10**6,
                period=dt(minutes=2),
                description="latency (μs): {{ pointValue }}",
            ).add_flap_detector(flap_detector_params)

            solomon.get_sensor("quoter_server.request.get_quota_state.status.ok").create_threshold_check(
                aggregation=alert_pb2.AVG,
                predicate=alert_pb2.LT,
                threshold=30,
                period=dt(minutes=2),
                description="rpm: {{ pointValue }}",
            ).add_flap_detector(flap_detector_params)

            solomon.get_sensor("state_updater.state.production.status.full").create_threshold_check(
                aggregation=alert_pb2.AVG,
                predicate=alert_pb2.GT,
                threshold=0,
                period=dt(minutes=2),
                description="events per minute: {{ pointValue }}",
            ).add_flap_detector(flap_detector_params).add_nodata_mode(consts.NoDataMode.force_ok).add_phone_escalation()

            solomon.get_sensor("state_updater.state.production.errors").create_threshold_check(
                aggregation=alert_pb2.AVG,
                predicate=alert_pb2.GT,
                threshold=0,
                period=dt(minutes=2),
                description="errors per minute: {{ pointValue }}",
            ).add_flap_detector(flap_detector_params).add_nodata_mode(consts.NoDataMode.force_ok)


def add_rt_duid_uploader_checks(juggler_check_generator, yasm):
    upload_thresholds = {
        environment.PRODUCTION: 300,
        environment.TESTING: 0,
    }

    yasm.add_devops(service="CM Duid Uploader", prj_tag="crypta-cm-rt-duid-uploader-to-cm")

    for dc, env, nanny_service, service in [
        ("sas", environment.TESTING, "test_crypta_cm_duploader_cm_sas", "rt_duid_uploader_to_cm"),
        ("sas", environment.PRODUCTION, "prod_crypta_cm_duploader_cm_sas", "rt_duid_uploader_to_cm"),
        ("vla", environment.PRODUCTION, "prod_crypta_cm_duploader_cm_vla", "rt_duid_uploader_to_cm"),
    ]:
        flap_detector_params = FlapDetectorParams(dt(minutes=10), dt(minutes=30))

        juggler = juggler_check_generator.clone(
            child_group=nanny_service,
            child_group_type=consts.GroupType.nanny,
            escalation=False,
            warn_limit=0,
            crit_limit=100,
        )
        juggler.icmpping().add_flap_detector(flap_detector_params)

        solomon = solomon_check_generator.DcSolomonCheckGenerator(
            juggler,
            project=PROJECT,
            service=service,
            cluster=env,
            dc=dc,
        )
        solomon.get_sensor("parser.errors.bad_command").create_threshold_check(
            aggregation=alert_pb2.MAX,
            predicate=alert_pb2.GT,
            threshold=0,
            period=dt(minutes=5),
            description="parsing errors (msgs): {{ pointValue }}",
        ).add_nodata_mode(consts.NoDataMode.force_ok)

        solomon.create_sensor_aggregation_check(
            service="matches_uploaded",
            period=dt(minutes=5),
            sensor_glob="uploader.upload.success",
            predicate=alert_pb2.LTE,
            threshold=upload_thresholds[env],
            description="Total uploaded : {{expression.total}} < {{expression.threshold}}",
        ).add_flap_detector(flap_detector_params)

        solomon.create_hist_check(
            service="upload_retries",
            period=dt(minutes=5),
            labels_bins={"sensor": "uploader.upload.retries.b0"},
            labels_all={"sensor": "uploader.upload.retries.*"},
            threshold=0.9,
            predicate=alert_pb2.LT,
            description="Uploaded with zero retries : {{expression.percentile}} < {{expression.threshold}}",
        ).add_flap_detector(flap_detector_params).remove_phone_escalation()

        solomon.create_logbroker_lag_check(
            consumer="crypta/{env}/cm/{service}_consumer".format(env=env[:4], service=service),
            topic="crypta/prod/graph/fingerprint-matching",
            threshold=10 * 1000,
            period=dt(minutes=5),
        ).add_flap_detector(flap_detector_params)


def add_db_stats(juggler):
    # 31 days are normal right after expiration process has ended
    # 32 days would trigger while expiration process is active
    days_ago_threshold = 33

    solomon = solomon_alert_registry.SolomonAlertRegistry()
    juggler.add_subregistry(solomon)

    def get_days_ago_sensor(days_ago_threshold):
        return "|".join(str(days_ago) for days_ago in db_stats_lib.DAYS_AGO_BUCKETS if days_ago >= days_ago_threshold)

    for dc, env, yt_proxy in [
        ("sas", environment.PRODUCTION, YtProxy.seneca_sas),
        ("vla", environment.PRODUCTION, YtProxy.seneca_vla),
        ("vla", environment.TESTING, YtProxy.zeno),
        ("sas", environment.TESTING, YtProxy.seneca_sas),
    ]:
        host = "{}.{}.crypta_cm.db_stats".format(dc, env)
        raw_host = "{}.raw".format(host)
        service = "touched_days_ago"

        # TODO(CRYPTAYT-3771): use check generator
        solomon.add_solomon_alert(solomon_alert_utils.create_expression_alert(
            project_id=PROJECT,
            name="{}.touched_days_ago".format(host),
            period=dt(hours=6),
            program="""
            let counts_sum = avg(
                group_lines("sum",
                    group_by_time(1m, "sum", {{
                        project="{project}",
                        cluster="{env}",
                        dc="{dc}",
                        service="db_stats",
                        sensor="touch_count",
                        days_ago=~"{days_ago_regexp}"
                    }})
                )
            );
            let threshold = 1;

            warn_if(counts_sum > threshold);
            """.format(
                project=PROJECT,
                env=env,
                dc=dc,
                days_ago_regexp=get_days_ago_sensor(days_ago_threshold),
            ),
            group_by_labels=[],
            juggler_service=service,
            juggler_host=raw_host,
            juggler_description=(
                "records touched {days_ago_threshold} days ago or earlier : "
                "{{{{ expression.counts_sum }}}} > {{{{ expression.threshold }}}}"
            ).format(days_ago_threshold=days_ago_threshold),
        ))

        juggler.any(service)\
            .add_flap_detector(FlapDetectorParams(dt(minutes=10), dt(minutes=30)))\
            .add_yt_dependencies([yt_proxy])\
            .set_host(host)\
            .set_child(raw_host)
