import argparse

from crypta.cm.tools.bin.dyn_tools.lib import helpers
from crypta.lib.python.yt import (
    tm_utils,
    yt_helpers,
)
from crypta.lib.python.yt.dyntable_utils import add_table_replica


def add_args_subparser(subparsers):
    parser = subparsers.add_parser(
        "add-replica",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        description=(
            "Add exact copy of replica to master table\n"
            "Examples:\n"
            "  cm-tools add-replica --production --dst-dc vla  # restore missing replica\n"
            "  cm-tools add-replica --production --dst-dc vla --dst-table //home/bigb/crypta_replicas/replica  # restore replica for a third party\n"
        )
    )

    parser.set_defaults(function=main)

    helpers.add_config_args(parser)

    helpers.add_cluster_args(parser, "--src-dc", "--src-cluster")
    helpers.add_cluster_args(parser, "--dst-dc", "--dst-cluster")

    parser.add_argument("--src-table", help="Source table")
    parser.add_argument("--dst-table", help="Destination table")
    parser.add_argument("--dry-run", action="store_true", help="Only show prospective changes")
    parser.add_argument("--temp-prefix", default="//tmp/", help="directory for temp files")
    parser.add_argument("--yt-pool", default="default", help="YT pool")
    parser.add_argument("--force", action="store_true", default=False, help="overwrite destination path")


def main(
    config,
    src_dc,
    src_cluster,
    dst_dc,
    dst_cluster,
    src_table,
    dst_table,
    dry_run,
    temp_prefix,
    yt_pool,
    force,
):
    config = helpers.get_config(config)

    master_cluster = helpers.get_master_cluster(config)
    master_path = helpers.get_master_path(config)

    src_cluster = helpers.select_replica_cluster_name(src_dc, src_cluster, config)
    dst_cluster = helpers.select_replica_cluster_name(dst_dc, dst_cluster, config)

    src_path = helpers.select_replica_path(src_table, config)
    dst_path = helpers.select_replica_path(dst_table, config)

    replica_attrs, replication_attrs = helpers.get_cluster_attrs(dst_cluster, config)

    print(
        f"Adding table replica:\n"
        f"  master cluster:        {master_cluster}\n"
        f"  master path:           {master_path}\n"
        f"  src replica cluster:   {src_cluster}\n"
        f"  src replica path:      {src_path}\n"
        f"  dst cluster:           {dst_cluster}\n"
        f"  dst path:              {dst_path}\n"
        f"  dst replica attrs:     {replica_attrs}\n"
        f"  dst replication attrs: {replication_attrs}\n"
    )

    if dry_run:
        print("NO CHANGES will by applied due to --dry-run")
        success = True
    else:
        yt_token = helpers.get_robot_crypta_token(config["environment_type"])

        success = add_table_replica.add_table_replica(
            master_client=yt_helpers.get_yt_client(master_cluster, yt_token=yt_token),
            master_table=master_path,
            src_client=yt_helpers.get_yt_client(src_cluster, yt_token=yt_token),
            src_table=src_path,
            dst_client=yt_helpers.get_yt_client(dst_cluster, yt_token=yt_token),
            dst_table=dst_path,
            temp_prefix=temp_prefix,
            yt_pool=yt_pool,
            force=force,
            tm_client=tm_utils.get_client(yt_token),
            replica_attrs=replica_attrs,
            replication_attrs=replication_attrs,
        )

    return 0 if success else 1
