import os

import yaml

from crypta.lib.python import templater

CONFIG_DIR = "/etc/crypta/cm-tools/configs"
PRODUCTION_CONFIG = os.path.join(CONFIG_DIR, "production.yaml")
TESTING_CONFIG = os.path.join(CONFIG_DIR, "testing.yaml")

SECRETS_DIR = "/etc/crypta/secrets/cm-tools"

MASTER = "master"
REPLICA = "replica"
PATH = "path"


def get_config(config_path):
    with open(config_path) as f:
        return yaml.safe_load(templater.render_template(f.read(), strict=True))


def get_token(token_path):
    with open(token_path) as f:
        return f.read().rstrip()


def get_robot_crypta_token(environment_type):
    if environment_type == "production":
        user_name = "robot-crypta"
    elif environment_type == "testing":
        user_name = "robot-crypta-testing"
    elif environment_type == "qa":
        return "XXX"
    else:
        raise Exception("Unknown environment_type = {}".format(environment_type))

    return get_token(os.path.join(SECRETS_DIR, user_name))


def get_master_cluster(config):
    return config[MASTER]["cluster"]


def get_master_path(config):
    return config[MASTER][PATH]


def get_replica_path(config):
    return config[REPLICA][PATH]


def select_replica_cluster_name(dc, cluster, config):
    if cluster is not None:
        name = cluster
    else:
        name = "seneca-{}".format(dc)

    known_clusters = config[REPLICA]["clusters"]
    if name not in known_clusters:
        raise Exception("Unknown cluster {}. List of known clusters: {}".format(name, known_clusters))

    return name


def select_replica_path(path, config):
    if path is not None:
        return path
    return get_replica_path(config)


def get_cluster_attrs(cluster, config):
    return tuple(config[REPLICA]["clusters"][cluster].get(key, {}) for key in ("replica_attrs", "replication_attrs"))


def add_config_args(argument_parser):
    group = argument_parser.add_mutually_exclusive_group(required=True)
    group.add_argument("--production", action="store_const", dest="config", const=PRODUCTION_CONFIG, help="Use production configuration file")
    group.add_argument("--testing", action="store_const", dest="config", const=TESTING_CONFIG, help="Use testing configuration file")
    group.add_argument("--config", help="Master and replicas configuration file")


def add_cluster_args(argument_parser, dc_arg, replica_cluster_arg):
    cluster_group = argument_parser.add_mutually_exclusive_group(required=True)
    cluster_group.add_argument(dc_arg, help="Datacenter: sas, vla, man. Will be interpreted as a corresponding seneca-*")
    cluster_group.add_argument(replica_cluster_arg, help="YT cluster name")
