import json

import luigi
import yt.wrapper as yt

from lib import phone_parser
from lib.luigi import yt_luigi
from rtcconf import config
from utils import mr_utils as mr
from v2.soup import soup_config
from v2.soup.soup_tables import SoupDumpTable


def _map_people_search(rec):
    data = json.loads(rec['value'])
    network = data['network']
    profile_id = data.get('profileID')

    if profile_id:
        login = data.get('login', None)
        gender = data.get('gender', None)

        birth_date = data.get('date', None)
        if birth_date:
            birth_date = '%s-%s-%s' % (birth_date[:4], birth_date[4:6], birth_date[6:8])
        date_type = data.get('dateType', None)

        emails = data.get('email', [])

        phones_raw = data.get('mobile', [])
        phone_regions = []
        phones = []
        for phone_raw in phones_raw:
            number_and_region = phone_parser.parse_phone_number(phone_raw)
            if number_and_region:
                phone, region = number_and_region
                phones.append(phone)
                phone_regions.append(region)

        out_rec = {'login': login, 'gender': gender,
                   'birth_date': birth_date, 'date_type': date_type,
                   'emails': emails, 'phones': phones, 'phone_regions': phone_regions,
                   'data': data, 'profile_source': 'people_search'}

        if network == 'vk':
            profile_id = profile_id.replace('id', '')
            out_rec['id_value'] = profile_id
            out_rec['id_type'] = config.ID_TYPE_VKCOM
            yield out_rec
            for phone in phones:
                yield SoupDumpTable.make_rec(
                    phone,
                    profile_id,
                    soup_config.phone_pridvk_psearch,
                    [],
                    4
                )
            for email in emails:
                yield SoupDumpTable.make_rec(
                    email,
                    profile_id,
                    soup_config.email_pridvk_psearch,
                    [],
                    5
                )
        elif network == 'ok':
            profile_id = profile_id.replace('profile/', '')
            out_rec['id_value'] = profile_id
            out_rec['id_type'] = config.ID_TYPE_OKRU
            out_rec['@table_index'] = 1
            yield out_rec
            for phone in phones:
                yield SoupDumpTable.make_rec(
                    phone,
                    profile_id,
                    soup_config.phone_pridok_psearch,
                    [],
                    6
                )
            for email in emails:
                yield SoupDumpTable.make_rec(
                    email,
                    profile_id,
                    soup_config.email_pridok_psearch,
                    [],
                    7
                )
        else:
            rec['@table_index'] = 2
            yield rec
            # dont yield to soup, table empty

    else:
        rec['@table_index'] = 3
        yield rec
        # dont yield to soup, there is no profile_id


class ImportPeopleSearch(yt_luigi.BaseYtTask):
    date = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(ImportPeopleSearch, self).__init__(*args, **kwargs)
        self.phone_pridvk_psearch = SoupDumpTable(soup_config.phone_pridvk_psearch, self.date)
        self.email_pridvk_psearch = SoupDumpTable(soup_config.email_pridvk_psearch, self.date)
        self.phone_pridok_psearch = SoupDumpTable(soup_config.phone_pridok_psearch, self.date)
        self.email_pridok_psearch = SoupDumpTable(soup_config.email_pridok_psearch, self.date)

    def input_folders(self):
        return {
            'people_search': '//home/freshness/crypta/',
            'people_search_dump': config.GRAPH_YT_DICTS_FOLDER + 'users_birth_dates_for_crypta'
        }

    def output_folders(self):
        return {
            'people_search_dict': config.GRAPH_YT_DICTS_FOLDER + 'people_search/',
        }

    def requires(self):
        return [
            yt_luigi.ExternalInput(self.in_f('people_search') + 'users_birth_dates'),
            yt_luigi.ExternalInput(self.in_f('people_search_dump'))
        ]

    def run(self):
        workdir = self.out_f('people_search_dict')
        mr.mkdir(workdir)
        with yt.Transaction() as tr:
            phone_parser.init()

            soup_tmp_tables = [
                self.phone_pridvk_psearch.create(tr),
                self.email_pridvk_psearch.create(tr),
                self.phone_pridok_psearch.create(tr),
                self.email_pridok_psearch.create(tr)
            ]

            yt.run_map(
                _map_people_search,
                [
                    self.in_f('people_search') + 'users_birth_dates',
                    self.in_f('people_search_dump')
                ],
                [
                    workdir + 'vk',
                    workdir + 'ok',
                    workdir + 'other',
                    workdir + 'login_only',
                ] + soup_tmp_tables,
            )

            SoupDumpTable.finalize_all([
                self.phone_pridvk_psearch,
                self.email_pridvk_psearch,
                self.phone_pridok_psearch,
                self.email_pridok_psearch
            ], tr)

            mr.sort_all([
                workdir + 'vk',
                workdir + 'ok'
            ], sort_by=['id_value', 'id_type'])

            mr.set_generate_date(workdir + 'vk', self.date)
            mr.set_generate_date(workdir + 'ok', self.date)

        return

    def output(self):
        soup_targets = [t.as_target() for t in [
            self.phone_pridvk_psearch,
            self.email_pridvk_psearch,
            self.phone_pridok_psearch,
            self.email_pridok_psearch
        ]]

        workdir = self.out_f('people_search_dict')
        return [yt_luigi.YtDateTarget(workdir + 'vk', self.date),
                yt_luigi.YtDateTarget(workdir + 'ok', self.date)] + soup_targets
