# coding=utf-8

"""
Partners table schema import tool
"""

import white

from v2 import ids
from v2.soup import soup_config
from v2.soup import soup_edge_type
from v2.soup.soup_tables import SoupDumpTable

# How many emails/phones or other ID should be per partner record, or it will be declined as
# a whole (organisation contacts case e.t.c.)
MAX_IDS_BEHIND_PARTNER_EXTERNAL_ID = 3

WHITE_TABLE = {
    ids.EMAIL: white.WhiteEmail,
    ids.PHONE: white.WhitePhone,
    'date': white.WhiteDate,
}

def _get_white(key):
    return WHITE_TABLE[key]()


# Import
def _validate_external_id_cardinality(rec):
    emails_ok = len(rec.get("emails", []) or []) <= MAX_IDS_BEHIND_PARTNER_EXTERNAL_ID
    phones_ok = len(rec.get("phones", []) or []) <= MAX_IDS_BEHIND_PARTNER_EXTERNAL_ID
    return emails_ok and phones_ok


def _parse_email_dict(dct):
    if isinstance(dct, dict) and dct:
        wp = _get_white(ids.EMAIL)
        if dct.get('email'):
            return wp.process(dct['email'])
        elif dct.get('id_value'):
            return wp.process(dct['id_value'])


def _parse_phone_dict(dct):
    if isinstance(dct, dict) and dct:
        wp = _get_white(ids.PHONE)
        if dct.get('phone'):
            return wp.process(dct['phone'])
        elif dct.get('id_value'):
            return wp.process(dct['id_value'])


def _white_check_if_hash(white_result):
    """
    :param white_result: white processing result as is
    :return: tuple(value, is_hash_boolean)
    """
    hashed_value = white_result['hash']
    if hashed_value:  # means validation is ok
        if hashed_value == white_result.get('orig'):
            # was originally hash
            return hashed_value, True
        else:
            validated_clean_value = white_result.get('value')
            if validated_clean_value:
                return validated_clean_value, False
            else:
                return None, None
    else:
        return None, None


def map_partners_audience_dumps(rec, in_partner_source_by_index, out_soup_edge_types_index):
    """Adapter for Audience data.

    https://wiki.yandex-team.ru/x-products/RealTimeTriggers/Arxitektura/API/audience/

    Example:
    {
      "external_id":"1",
      "impulses":[
        {
          "impulse": "RKO",
          "delete":"2007-05-01T23:20:50.52Z"
        },
        {
          "impulse": "platimun"
        }
      ],
      "phones":[
        {
          "id_value":"b356af14c869ed7e31e185cbf075b89b",
          "phone":"79160444381"
        },
        {
          "id_value":"50710784ab8398162e5772f8996eea5a",
          "phone":"79137172777",
          "delete":"2007-05-01T23:20:50.52Z"
        }

      ],
      "emails":[
        {
          "id_value":"58245df017691357bcff7ef02057a7dd",
          "email":"d-sun-d@yandex.ru"
        },
        {
          "id_value":"934fba84a48a90feddcafb7698569ce2",
          "email":"riabchenkoaa@gmail.com"
        },
        {
          "id_value":"b6ecfae9a14764a225798c4f4edf4820"
        }
      ],
      "birthdate":{
        "year":1989,
        "month":6
      },
      "gender":"M",
      "add_date":"2017-05-01T23:20:50.52Z",
      "other_fields":[
        {
              "field_name":"vk_url",
              "field_value":"https://vk.com/dsund"
        },
        {
              "field_name":"name",
              "field_value":"Александр"
        }
     ],
     "yandex_can_use":true
    }
    """

    def parse_emails_phones(rec):
        white_emails = filter(bool, [_parse_email_dict(dct) for dct in rec.get("emails", []) or []])
        white_phones = filter(bool, [_parse_phone_dict(dct) for dct in rec.get("phones", []) or []])

        for email_white in white_emails:
            id_value, is_hash = _white_check_if_hash(email_white)
            if id_value:
                if is_hash:
                    id_type = ids.EMAIL + ids.MD5_POSTFIX
                else:
                    id_type = ids.EMAIL

                yield {'id': id_value, 'id_type': id_type}

        for phone_white in white_phones:
            id_value, is_hash = _white_check_if_hash(phone_white)
            if id_value:
                if is_hash:
                    id_type = ids.PHONE + ids.MD5_POSTFIX
                else:
                    id_type = ids.PHONE

                yield {'id': id_value, 'id_type': id_type}

    partner_source = in_partner_source_by_index[rec['@table_index']]

    err_table_index = len(out_soup_edge_types_index)
    ok = _validate_external_id_cardinality(rec)
    if not ok:
        rec['source_type'] = partner_source
        rec['@table_index'] = err_table_index
        yield rec
        return

    dates = []
    rec_date = rec.get("add_date")
    if rec_date:
        date = _get_white('date').process(rec_date)['value']
        if date:
            dates.append(date)


    external_id_prefix = partner_source.replace('partner_', '')
    external_id = rec['external_id']
    partner_id = '{}_{}'.format(external_id_prefix, external_id)

    for parsed in parse_emails_phones(rec):
        id_type = parsed['id_type']
        edge_type = soup_edge_type.get_edge_type(
            id1_type=ids.PARTNER_RECORD_ID,
            id2_type=id_type,
            source=partner_source,
            log_source=soup_config.LOG_SOURCE_PARTNERS_AUDIENCE
        )
        yield SoupDumpTable.make_rec(
            partner_id, parsed['id'], edge_type, dates,
            table_index=out_soup_edge_types_index[edge_type]
        )
