import os
from collections import namedtuple
from functools import partial

import luigi
import yt.wrapper as yt

from data_imports.day_aggregate import reduce_yuid_log_events_day, finalize_yuid_with_x_day_tables
from lib.luigi import yt_luigi
from rtcconf import config
from utils import mr_utils as mr
from v2.soup import soup_config
from v2.soup.soup_tables import SoupDumpTable


def reduce_find_max(key, recs, dt):
    ValueWithCount = namedtuple('ValueWithCount', 'value count')
    max_id_login = {'id': ValueWithCount(None, 0), 'username': ValueWithCount(None, 0)}
    for rec in recs:
        count = int(rec.get('message_count', '0'))
        rec_type = rec.get('type', '')
        if count > max_id_login[rec_type].count:
            max_id_login[rec_type] = ValueWithCount(str(rec.get('value', '')), count)
    for k in max_id_login:
        if max_id_login[k].value:
            puid = str(key['uid'])

            if k == 'id':
                instagram_id = max_id_login[k].value
                yield {'id_value': instagram_id, 'puid': puid, 'id_date': dt,
                       '@table_index': 0}
                yield SoupDumpTable.make_rec(puid, instagram_id, soup_config.puid_insta_id_maildump, [], 2)

            elif k == 'login':
                instagram_login = max_id_login[k].value
                yield {'id_value': instagram_login, 'puid': puid, 'id_date': dt,
                       '@table_index': 1}
                yield SoupDumpTable.make_rec(puid, instagram_login, soup_config.puid_insta_login_maildump, [], 3)


class ImportInstagramDayTask(yt_luigi.BaseYtTask):
    date = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(ImportInstagramDayTask, self).__init__(*args, **kwargs)
        self.puid_insta_id_maildump = SoupDumpTable(soup_config.puid_insta_id_maildump, self.date)
        self.puid_insta_login_maildump = SoupDumpTable(soup_config.puid_insta_login_maildump, self.date)

    def input_folders(self):
        return {
            'instagram_log': config.INSTAGRAM_INPUT_FOLDER,
        }

    def output_folders(self):
        return {
            'raw': config.YT_OUTPUT_FOLDER + self.date + '/yuid_raw/',
        }

    def requires(self):
        return [] # yt_luigi.ExternalInput(self.in_f('instagram_log') + self.date.replace('-', ''))

    def run(self):
        mr.mkdir(self.out_f('raw'))
        src = self.in_f('instagram_log') + self.date.replace('-', '')
        src_tmp = os.path.join(self.out_f('raw'), 'instagram_log_tmp')
        insta_id_out = os.path.join(self.out_f('raw'),
                                    'puid_with_' + config.ID_TYPE_INSTAGRAM_ID + '_' +
                                    config.ID_SOURCE_TYPE_INSTAGRAM_POCHTA)
        insta_login_out = os.path.join(self.out_f('raw'),
                                    'puid_with_' + config.ID_TYPE_INSTAGRAM_LOGIN + '_' +
                                    config.ID_SOURCE_TYPE_INSTAGRAM_POCHTA)
        insta_id_tmp = insta_id_out + '_tmp'
        insta_login_tmp = insta_login_out + '_tmp'

        with yt.Transaction() as tr:

            yt.create('table', insta_id_out, ignore_existing=True)
            yt.create('table', insta_login_out, ignore_existing=True)


            if yt.exists(src) and not yt.is_empty(src):

                yt.run_sort(src, src_tmp,  sort_by='uid')
                yt.run_reduce(partial(reduce_find_max, dt=self.date),
                              src_tmp,
                              [insta_id_tmp, insta_login_tmp,
                               self.puid_insta_id_maildump.create(tr),
                               self.puid_insta_login_maildump.create(tr)
                               ],
                              reduce_by=['uid'])

                mr.sort_all([insta_id_tmp, insta_login_tmp], sort_by='puid')

                yt.run_reduce(partial(reduce_yuid_log_events_day, dt=self.date, key_col='puid',
                                      id_type=config.ID_TYPE_INSTAGRAM_ID,
                                      source_type=config.ID_SOURCE_TYPE_INSTAGRAM_POCHTA),
                              insta_id_tmp,
                              insta_id_out,
                              reduce_by='puid')

                yt.run_reduce(partial(reduce_yuid_log_events_day, dt=self.date, key_col='puid',
                                      id_type=config.ID_TYPE_INSTAGRAM_LOGIN,
                                      source_type=config.ID_SOURCE_TYPE_INSTAGRAM_POCHTA),
                              insta_login_tmp,
                              insta_login_out,
                              reduce_by='puid')

                finalize_yuid_with_x_day_tables([insta_id_out, insta_login_out])

                yt.remove(src_tmp)
                yt.remove(insta_id_tmp)
                yt.remove(insta_login_tmp)

            else:
                self.puid_insta_id_maildump.create(tr)
                self.puid_insta_login_maildump.create(tr)

            self.puid_insta_id_maildump.finalize(tr)
            self.puid_insta_login_maildump.finalize(tr)


    def output(self):
        return [yt_luigi.YtTarget(self.out_f('raw') + 'puid_with_' + config.ID_TYPE_INSTAGRAM_ID + '_' +
                                  config.ID_SOURCE_TYPE_INSTAGRAM_POCHTA, allow_empty=True),
                yt_luigi.YtTarget(self.out_f('raw') + 'puid_with_' + config.ID_TYPE_INSTAGRAM_LOGIN + '_' +
                                  config.ID_SOURCE_TYPE_INSTAGRAM_POCHTA, allow_empty=True)] + \
               [t.as_target() for t in [self.puid_insta_id_maildump, self.puid_insta_login_maildump]]
