# coding=utf-8
import luigi
import yt.wrapper as yt

from lib.luigi import yt_luigi
from rtcconf import config
from utils import mr_utils as mr
from v2.soup import soup_config
from v2.soup.soup_tables import SoupDumpTable


def mapper_mail_vkid_puid(rec):
    puid = rec.get('uid', None)
    vk_id = rec.get('vk_id', None)
    if puid and vk_id:
        yield {
            'puid': puid,
            'vk_id': vk_id
        }


def reducer_mail_vkid_puid(keys, _):
    yield SoupDumpTable.make_rec(
        str(keys['puid']),
        str(keys['vk_id']),
        soup_config.puid_vkcom_maildump,
        [],
        0
    )
    yield {
        'uid': str(keys['puid']),
        'vk_id': str(keys['vk_id']),
        '@table_index': 1
    }


def join_reducer(_, records):
    input_record = None
    joined_record = None
    for record in records:
        if record['@table_index'] == 0:
            input_record = record
        elif record['@table_index'] == 1:
            joined_record = record
    if input_record and joined_record and \
            'vk_id' in input_record and \
            'uid' in input_record:
        login = joined_record.get('login')
        if login:
            email = joined_record['login']
            if '@' not in email:
                email += '@yandex.ru'
            yield {
                "vk_id": input_record['vk_id'],
                "puid": input_record['uid'],
                "email": email,
            }


class ImportVkidPuidMailDump(yt_luigi.BaseYtTask):
    date = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(ImportVkidPuidMailDump, self).__init__(*args, **kwargs)
        self.soup_table = SoupDumpTable(soup_config.puid_vkcom_maildump, self.date)

    def input_folders(self):
        return {
            'export': config.VKID_PUID_DMP_FOLDER,
            'passport': config.STATBOX_SQL_PASSPORT
        }

    def output_folders(self):
        return {
            'dict': config.GRAPH_YT_DICTS_FOLDER,
        }

    @property
    def output_table(self):
        return self.out_f('dict') + 'vkids_with_emails_from_mail_source'

    def output(self):
        soup_out_tables = [self.soup_table.as_target()]
        return soup_out_tables + [
            yt_luigi.YtDateTarget(self.output_table, self.date)
        ]

    def run(self):

        dump_table = self.in_f('export') + 'vk_ids'

        with yt.Transaction() as transaction,\
            yt.TempTable() as formatted_table:

            yt.run_map_reduce(
                mapper_mail_vkid_puid,
                reducer_mail_vkid_puid,
                dump_table,
                [self.soup_table.create(transaction), formatted_table],
                reduce_by=['puid', 'vk_id']
            )

            yt.run_sort(formatted_table, sort_by='uid')

            mr.create_table_with_schema(
                self.output_table,
                {
                    "vk_id": 'string',
                    "puid": 'string',
                    "email": 'string'
                },
                transaction,
                recreate_if_exists=True
            )

            passport_table = yt.TablePath(
                mr.get_last_table(self.in_f('passport')), attributes={'foreign': True}
            )

            yt.run_join_reduce(
                join_reducer,
                [formatted_table, passport_table],
                self.output_table,
                reduce_by='uid',
                join_by='uid',
            )

            yt.run_merge(self.output_table, self.output_table, spec={'combine_chunks': True})

            self.soup_table.finalize(transaction)

            for out_target in self.output():
                mr.set_generate_date(out_target.table, self.date)
