import logging
from functools import partial

import luigi
import white
import yt.wrapper as yt

import social_auth_providers
from data_imports.day_aggregate import reduce_yuid_log_events_day, finalize_yuid_with_x_day_tables
from lib.luigi import yt_luigi
from rtcconf import config
from utils import mr_utils as mr
from utils import utils
from v2 import ids
from v2.soup import soup_config, soup_edge_type
from v2.soup.soup_tables import SoupDumpTable

white_phone = white.WhitePhone()


def parse_social_id(rec, provider_mapping):
    provider_id = rec['provider_id']
    if provider_id in provider_mapping:
        social_id = rec['userid']
        social_id_type = provider_mapping[provider_id]

        # https://st.yandex-team.ru/CRYPTR-156#1510906029000
        if social_id_type == ids.FB_ID and ":" in social_id:
            return None, None

        return social_id, social_id_type
    else:
        return None, None


def map_passport_social(rec, provider_mapping, social_edge_type_indexes):
    puid = rec['uid']
    date = rec['created'][:10]

    email = utils.norm_email(rec.get('email'))
    phone = rec.get('phone')
    phone_hash = None

    if email:
        yield {'puid': puid, 'id_value': email, '@table_index': 0}

    if phone:
        processed_phone = white_phone.process(phone)
        # processed_phone = {'hash': phone, 'prefix': None}
        phone_hash = processed_phone['hash']
        phone_prefix = processed_phone['prefix']
        if phone_hash:
            yield {'puid': puid, 'id_value': phone_hash, 'id_prefix': phone_prefix, '@table_index': 1}

    def to_soup_rec(id1, id1_type, id2, id2_type):

        if (id1_type, id2_type) in social_edge_type_indexes:
            edge_type = soup_edge_type.get_edge_type(id1_type, id2_type,
                                                     soup_config.ID_SOURCE_TYPE_SOCIAL_AUTH,
                                                     soup_config.LOG_SOURCE_SOCIALDB)

            table_index = 2 + social_edge_type_indexes[(id1_type, id2_type)]  # shift for non soup tables

            return SoupDumpTable.make_rec(
                id1, id2, edge_type, [date],
                table_index
            )
        else:
            not_parsed_index = 2 + len(social_edge_type_indexes)
            return {"id1": id1, "id1_type": id1_type, "id2": id2, "id2_type": id2_type,
                    '@table_index': not_parsed_index}

    social_id, social_id_type = parse_social_id(rec, provider_mapping)

    if social_id and social_id_type:

        yield to_soup_rec(puid, ids.PUID, social_id, social_id_type)

        # for supported social ids put email and phone to social id
        if email:
            yield to_soup_rec(social_id, social_id_type, email, ids.EMAIL)
        if phone_hash:
            yield to_soup_rec(social_id, social_id_type, phone_hash, ids.PHONE + ids.MD5_POSTFIX)

    else:
        # for unsupported social ids put email and phone directly to puid
        if email:
            yield to_soup_rec(puid, ids.PUID, email, ids.EMAIL)
        if phone_hash:
            yield to_soup_rec(puid, ids.PUID, phone_hash, ids.PHONE + ids.MD5_POSTFIX)


class ImportSocialAuthDump(yt_luigi.BaseYtTask):
    date = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(ImportSocialAuthDump, self).__init__(*args, **kwargs)
        soup_edge_types = [soup_config.social_auth_puid_email, soup_config.social_auth_puid_phone] + \
                          soup_config.social_auth_social_id_pairs
        self.soup_storage_tables = [SoupDumpTable(t, self.date) for t in soup_edge_types]

    def input_folders(self):
        return {
            # 'passport_social': '//home/passport/production/socialism/crypta-dump/'
            'passport_social': config.PASSPORT_SOCIAL_FOLDER
        }

    def output_folders(self):
        return {
            # 'yuid_raw': '//home/crypta/team/artembelov/passport_social/yuid_raw/'
            'yuid_raw': config.GRAPH_YT_DICTS_FOLDER + 'yuid_raw/'

        }

    def requires(self):
        return yt_luigi.ExternalInput(self.in_f('passport_social') + self.date)

    def run(self):
        yuid_raw = self.out_f('yuid_raw')
        mr.mkdir(yuid_raw + 'social')

        with yt.Transaction() as tr:
            in_table = self.in_f('passport_social') + self.date

            edge_type_indexes = {(t.edge_type.id1_type, t.edge_type.id2_type): idx
                                 for idx, t
                                 in enumerate(self.soup_storage_tables)
                                 }
            yt.run_map(
                partial(map_passport_social,
                        provider_mapping=social_auth_providers.provider_id_to_id_type,
                        social_edge_type_indexes=edge_type_indexes
                        ),
                in_table,
                [
                    yuid_raw + 'social/puid_email',
                    yuid_raw + 'social/puid_phone'
                ] +
                [t.table_path() for t in self.soup_storage_tables] +
                [yuid_raw + 'social/not_parsed']
            )

            SoupDumpTable.finalize_all(self.soup_storage_tables, tr)

        # v1 compatibility
        out_email_table = yuid_raw + 'puid_with_' + config.ID_TYPE_EMAIL + '_' + config.ID_SOURCE_TYPE_SOCIAL
        out_phone_table = yuid_raw + 'puid_with_' + config.ID_TYPE_PHONE + '_' + config.ID_SOURCE_TYPE_SOCIAL

        utils.wait_all([
            yt.run_map_reduce(None, partial(reduce_yuid_log_events_day, dt=None,
                                            key_col=config.ID_TYPE_PUID,
                                            id_type=config.ID_TYPE_EMAIL,
                                            source_type=config.ID_SOURCE_TYPE_SOCIAL),
                              yuid_raw + 'social/puid_email',
                              out_email_table,
                              sort_by='puid', reduce_by='puid', sync=False),
            yt.run_map_reduce(None, partial(reduce_yuid_log_events_day, dt=None,
                                            key_col=config.ID_TYPE_PUID,
                                            id_type=config.ID_TYPE_PHONE,
                                            source_type=config.ID_SOURCE_TYPE_SOCIAL),
                              yuid_raw + 'social/puid_phone',
                              out_phone_table,
                              sort_by='puid', reduce_by='puid', sync=False)
        ])

        finalize_yuid_with_x_day_tables([out_email_table, out_phone_table], key_col=config.ID_TYPE_PUID)

        mr.set_generate_date(out_email_table, self.date)
        mr.set_generate_date(out_phone_table, self.date)

    def output(self):
        tables = [
            self.out_f('yuid_raw') + 'puid_with_' + config.ID_TYPE_EMAIL + '_' + config.ID_SOURCE_TYPE_SOCIAL,
            self.out_f('yuid_raw') + 'puid_with_' + config.ID_TYPE_PHONE + '_' + config.ID_SOURCE_TYPE_SOCIAL
        ]
        soup_targets = [t.as_target() for t in self.soup_storage_tables]
        return [yt_luigi.YtDateTarget(t, self.date) for t in tables] + soup_targets


if __name__ == '__main__':
    logging.basicConfig(level="INFO")
    yt.config["tabular_data_format"] = yt.YsonFormat(process_table_index=True)
    mr.mkdir(soup_config.SOUP_DUMPS_DIR)

    luigi.build([ImportSocialAuthDump('2018-07-20')], workers=1, local_scheduler=True)
