#!/usr/bin/python
# coding=utf-8

import os
import subprocess

import luigi
import yt.wrapper as yt
from lib.luigi import yt_luigi
from rtcconf import config

from data_imports.import_logs.import_rtb_log import ImportRTBLogDayTask

from v2.soup import soup_config
from v2.soup.soup_tables import SoupDailyLogTable

from utils.yql_utils import run_yql


class ImportAppMetrikaDayTask(yt_luigi.BaseYtTask):

    date = luigi.Parameter()
    run_date = luigi.Parameter()

    resources = {'import_app_metrika_lock': 1}
    priority = 1

    def __init__(self, *args, **kwargs):
        super(ImportAppMetrikaDayTask, self).__init__(*args, **kwargs)
        self.soup_log = SoupDailyLogTable(soup_config.LOG_SOURCE_MOBILE_METRIKA, self.date)

    def input_folders(self):
        return {
            'metrica': config.STATBOX_MMETRIC_FOLDER,
            'metrica_private': config.STATBOX_MMETRIC_PRIVATE_FOLDER,
            'mobile': config.YT_OUTPUT_FOLDER + self.date + '/mobile/',  # rtb logs,
        }

    def output_folders(self):
        return {
            'mobile': config.YT_OUTPUT_FOLDER + self.date + '/mobile/',
        }

    def requires(self):
        reqs = [
            yt_luigi.ExternalInput(self.in_f('metrica') + self.date),
            yt_luigi.ExternalInput(self.in_f('metrica_private') + self.date, allow_empty=True),
            ImportRTBLogDayTask(date=self.date, run_date=self.run_date), ]
        return reqs

    def run(self):
        out_f = self.out_f('mobile')
        self.soup_log.ensure_dir()

        run_yql('MetricaParser', dict(date=self.date), {
            'GRAPH_YT_OUTPUT_FOLDER': config.YT_OUTPUT_FOLDER,
            'INDEVICE_YT_FOLDER': config.INDEVICE_YT_FOLDER,
            'STATBOX_MMETRIC_FOLDER': config.STATBOX_MMETRIC_FOLDER,
            'STATBOX_MMETRIC_PRIVATE_FOLDER': config.STATBOX_MMETRIC_PRIVATE_FOLDER
        })

        self.soup_log.prepare_daily_tables_from_log()

    def output(self):
        if self.date == self.run_date:
            soup_out_tables = self.soup_log.daily_tables_targets()
            ip_dev_stream = yt_luigi.YtTarget(self.out_f('mobile') + 'ip_dev_stream')

            last_day_deps = soup_out_tables + [ip_dev_stream]
        else:
            last_day_deps = []  # don't want to keep it for 30 days, it's large

        return [
            yt_luigi.YtTarget(t) for t in [
                self.out_f('mobile') + 'uuid_info_yt',
                self.out_f('mobile') + 'dev_info_yt',
                self.out_f('mobile') + 'account_manager/am_log']] + last_day_deps


if __name__ == '__main__':
    yt.config.set_proxy(config.MR_SERVER)
    yt.config["tabular_data_format"] = yt.YsonFormat(process_table_index=True)

    task = ImportAppMetrikaDayTask('2017-07-12', '2017-07-12')

    # yt.run_map(map_rtb,
    #            '//home/crypta/production/state/graph/2017-07-12/mobile/rtb_log_apps',
    #            '//home/crypta/team/artembelov/metrika_rework/mobile/rtb_log_apps')
    #
    # yt.run_sort('//home/crypta/team/artembelov/metrika_rework/mobile/rtb_log_apps', sort_by=['device_id', 'ts'])

    luigi.build([task], workers=3, scheduler_port=8083)
