from textwrap import dedent

import luigi

from lib.luigi import yt_luigi
from rtcconf import config
from v2 import ids
from v2.soup import soup_config
from v2.soup.soup_tables import SoupDailyLogTable

from utils.yql_utils import run_yql

YQL = dedent('''
    PRAGMA yt.InferSchema; -- required for tests

    $source_type = '{source_type}';
    $log_source = '{log_source}';

    $uuid_name = '{uuid_name}';
    $yuid_name = '{yuid_name}';
    $avito_hash_name = '{avito_hash_name}';
    $phone_name = '{phone_name}';

    $input_table = [{input_table}];
    $output_table = [{output_table}];
    $rainbow_table = [{rainbow_table}];

    $date = '{date}';

    $avito_pageids = (
        '106253', -- Name=avito.ru
        '125616', -- Name=m.avito.ru
        '125618', -- Name=m.avito.ru
        '134783', -- Name=com.avito.android
        '134784', -- Name=ru.avito.app
        '145619', -- Name=avito.iOS
        '161308', -- Name=avito.ru
        '188382', -- Name=avito.ru
        '189903', -- Name=avito.ru
        '192620', -- Name=com.avito.android
        '192629', -- Name=ru.avito.app
        '207376', -- Name=com.avito.android
        '208411', -- Name=m.avito.ru
        '217153', -- Name=ru.avito.app
        '228614', -- Name=m.avito.ru
        '273712', -- Name=avito.ru
        '279179', -- Name=avito.ru
        '280196'  -- Name=avito.ru
    );

    $vals = (
        select
            uniqid,
            Url::GetCGIParam('http://fake/?' || queryargs, '245') as uuid,
            Url::GetCGIParam('http://fake/?' || queryargs, '726') as kw726,
            Url::GetCGIParam('http://fake/?' || queryargs, '727') as kw727
        from $input_table
        where pageid in $avito_pageids
    );

    $filtered_vals = (
        select uniqid, uuid, nvl(kw726, kw727) as avito_hash from $vals
        where uuid is not null or kw726 is not null or kw727 is not null
    );

    $grouped_vals = (
        select uniqid, uuid, avito_hash, count(*) as hits from $filtered_vals
        group by uniqid, uuid, avito_hash
    );

    $vals_with_phone = (
        select
            a.uniqid as uniqid,
            a.uuid as uuid,
            a.avito_hash as avito_hash,
            a.hits as hits,
            b.phone as phone
        from $grouped_vals as a
        left join $rainbow_table as b
          on a.avito_hash = b.avito_hash
    );

    insert into $output_table with truncate
    select
        nvl(uuid, uniqid) as id1,
        (case when uuid is not null then $uuid_name else $yuid_name end) as id1Type,
        nvl(phone, avito_hash) as id2,
        (case when phone is not null then $phone_name else $avito_hash_name end) as id2Type,
        $source_type as sourceType,
        $log_source as logSource,
        $date as date,
        null as dayActivity,
        hits as dayHits
    from $vals_with_phone
    where avito_hash is not null and (uuid is not null or uniqid is not null);
''')


class ImportAvito(luigi.Task):
    date = luigi.Parameter()
    run_date = luigi.Parameter()

    log_source = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(ImportAvito, self).__init__(*args, **kwargs)
        self.soup_log = SoupDailyLogTable(self.log_source, self.date)
        if self.log_source == soup_config.LOG_SOURCE_BS_HITLOG:
            self.log_table = config.LOG_FOLDERS['bs_hitlog'] + self.date
        elif self.log_source == soup_config.LOG_SOURCE_BS_RTBLOG:
            self.log_table = config.LOG_FOLDERS['bs_rtblog'] + self.date
        else:
            raise Exception('Unexpected log_source "{}" (expected {} or {})'.format(
                self.log_source, soup_config.LOG_SOURCE_BS_HITLOG, soup_config.LOG_SOURCE_BS_RTBLOG
            ))

    def requires(self):
        return yt_luigi.ExternalInput(self.log_table)

    def output(self):
        return self.soup_log.daily_tables_targets()

    def run(self):
        self.soup_log.ensure_dir()
        soup_tmp_table = self.soup_log.create()

        query = YQL.format(
            source_type=soup_config.ID_SOURCE_TYPE_AVITO,
            log_source=self.log_source,
            uuid_name=ids.UUID,
            yuid_name=ids.YANDEXUID,
            avito_hash_name=ids.AVITO_HASH,
            phone_name=ids.PHONE,
            input_table=self.log_table,
            output_table=soup_tmp_table,
            rainbow_table=config.GRAPH_YT_DICTS_FOLDER + 'avito-phone-rainbow',
            date=self.date
        )
        run_yql(query=query)

        self.soup_log.prepare_daily_tables_from_log()
