import sys
from functools import partial

import luigi
import yt.wrapper as yt

from data_imports.day_aggregate import reduce_device_log_events_day, finalize_device_yuid_day_tables
from lib.luigi import yt_luigi
from rtcconf import config
from utils import mr_utils as mr
from utils import utils
from v2.soup import soup_config
from v2.soup.soup_tables import SoupDailyLogTable


def map_oauth(rec):
    uuid = rec.get('uuid', '')
    device_id = utils.norm_id(rec.get('device_id', ''))
    uid = rec.get('uid', '')
    soup_table_index = 3

    try:
        ts = int(rec.get('unixtime', ''))
    except ValueError:
        rec['@table_index'] = 2
        yield rec
        return

    if uuid and uuid != '0' and uid:
        yield {config.ID_TYPE_UUID: uuid, config.ID_TYPE_PUID: uid, 'ts': -ts,
               'source': config.ID_SOURCE_TYPE_PASSPORT_OAUTH} # 'uuid' 'puid'
        yield SoupDailyLogTable.make_rec(
            uid,
            uuid,
            soup_config.puid_uuid_oauth,
            ts=-ts,
            table_index=soup_table_index
        )

    if device_id and uid:
        yield {'mmetric_devid': device_id, config.ID_TYPE_PUID: uid, 'ts': -ts,
               'source': config.ID_SOURCE_TYPE_PASSPORT_OAUTH, '@table_index': 1} # 'devid' 'puid'
        yield SoupDailyLogTable.make_rec(
            uid,
            device_id,
            soup_config.puid_devid_oauth,
            ts=-ts,
            table_index=soup_table_index
        )


def reduce_change_devid(key, recs):
    devid = ''
    for rec in recs:
        if rec['@table_index'] == 0:
            devid = rec.get('devid', '')
        elif rec['@table_index'] == 1 and devid:
            rec['devid'] = devid
            rec['@table_index'] = 0
            yield rec
        else:
            rec['@table_index'] = 1
            yield rec
            return


def import_oauth(oauth_log, dt, workdir, devid_raw_f, dict_f, soup_log):
    mr.mkdir(workdir)
    mr.mkdir(devid_raw_f)
    soup_log.ensure_dir()

    yt.run_map(map_oauth, oauth_log,
               [workdir + 'puid_uuid_oauth_tmp',
                workdir + 'puid_devid_oauth_tmp',
                workdir + 'oauth_bad',
                soup_log.create(),
                ])

    soup_log.prepare_daily_tables_from_log()

    utils.wait_all([
        yt.run_sort(workdir + 'puid_uuid_oauth_tmp',
                    sort_by=[config.ID_TYPE_PUID, config.ID_TYPE_UUID, 'ts'], sync=False),
        yt.run_sort(workdir + 'puid_devid_oauth_tmp',
                    sort_by=[config.ID_TYPE_PUID, 'mmetric_devid', 'ts'], sync=False),
    ])

    puid_uuid_oauth = devid_raw_f + 'puid_uuid_' + config.ID_SOURCE_TYPE_PASSPORT_OAUTH
    puid_devid_oauth = devid_raw_f + 'puid_devid_' + config.ID_SOURCE_TYPE_PASSPORT_OAUTH

    utils.wait_all([
        yt.run_reduce(partial(reduce_device_log_events_day, dt=dt,
                              source_type=config.ID_SOURCE_TYPE_PASSPORT_OAUTH),
                      workdir + 'puid_uuid_oauth_tmp',
                      puid_uuid_oauth,
                      sort_by=[config.ID_TYPE_PUID, config.ID_TYPE_UUID, 'ts'],
                      reduce_by=[config.ID_TYPE_PUID, config.ID_TYPE_UUID], sync=False),
        yt.run_reduce(partial(reduce_device_log_events_day, dt=dt,
                              source_type=config.ID_SOURCE_TYPE_PASSPORT_OAUTH),
                      workdir + 'puid_devid_oauth_tmp',
                      workdir + 'puid_devid_oauth_tmp_2',
                      sort_by=[config.ID_TYPE_PUID, 'mmetric_devid', 'ts'],
                      reduce_by=[config.ID_TYPE_PUID, 'mmetric_devid'], sync=False)
    ])

    utils.wait_all([
        yt.run_sort(dict_f + 'devid_hash', workdir + 'devid_hash', sort_by='mmetric_devid', sync=False),
        yt.run_sort(workdir + 'puid_devid_oauth_tmp_2', sort_by='mmetric_devid', sync=False)
    ])

    yt.run_reduce(reduce_change_devid, [workdir + 'devid_hash', workdir + 'puid_devid_oauth_tmp_2'],
                  [puid_devid_oauth,
                   puid_devid_oauth + '_nodevid'],
                  reduce_by='mmetric_devid')

    finalize_device_yuid_day_tables([
        puid_uuid_oauth,
        puid_devid_oauth
    ])

    mr.drop(workdir + 'puid_uuid_oauth_tmp')
    mr.drop(workdir + 'puid_devid_oauth_tmp')
    mr.drop(workdir + 'puid_devid_oauth_tmp_2')
    mr.drop(workdir + 'devid_hash')


class ImportOauthLogsDayTask(yt_luigi.BaseYtTask):
    date = luigi.Parameter()
    dict_date = luigi.Parameter()
    run_date = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(ImportOauthLogsDayTask, self).__init__(*args, **kwargs)
        self.soup_log = SoupDailyLogTable(soup_config.LOG_SOURCE_OAUTH, self.date)

    def input_folders(self):
        return {
            'statbox': config.LOG_FOLDERS['oauth'],
            'dict': config.GRAPH_YT_DICTS_FOLDER
        }

    def output_folders(self):
        return {
            'raw': config.INDEVICE_YT_FOLDER + self.date + '/perfect/devid_raw_day/',
            'logs': config.INDEVICE_YT_FOLDER + self.date + '/perfect/logs/',
        }

    def requires(self):
        from matching.device_matching.app_metrica import app_metrica_to_old_formats
        return [yt_luigi.ExternalInput(self.in_f('statbox') + self.date),
                app_metrica_to_old_formats.ConvertAppMetricaDictsToOldFormats(self.dict_date)]

    def run(self):

        mr.mkdir(self.out_f('raw'))
        mr.mkdir(self.out_f('logs'))

        oauth_log = config.LOG_FOLDERS['oauth'] + self.date

        import_oauth(oauth_log, self.date, self.out_f('logs'), self.out_f('raw'), self.in_f('dict'), self.soup_log)

    def output(self):

        if self.date == self.run_date:
            soup_out_tables = self.soup_log.daily_tables_targets()
        else:
            soup_out_tables = []

        out_date_folder = self.out_f('raw')
        return [yt_luigi.YtTarget(t) for t in
                [out_date_folder + 'puid_uuid_' + config.ID_SOURCE_TYPE_PASSPORT_OAUTH,
                 out_date_folder + 'puid_devid_' + config.ID_SOURCE_TYPE_PASSPORT_OAUTH]] + soup_out_tables


if __name__ == '__main__':
    yt.config.set_proxy(config.MR_SERVER)

    dt = sys.argv[1]

    import smart_runner
    smart_runner.run_isolated('//crypta/team/rodion/CRYPTAIS-911/', dt, ImportOauthLogsDayTask,
                              ImportOauthLogsDayTask, date=dt, dict_date=dt, run_date=dt)
