import sys
from functools import partial

import luigi
import yt.wrapper as yt

from data_imports.day_aggregate import reduce_device_log_events_day, finalize_device_yuid_day_tables
from lib.luigi import base_luigi_task
from lib.luigi import yt_luigi
from rtcconf import config
from utils import mr_utils as mr
from v2.soup import soup_config
from v2.soup.soup_tables import SoupDailyLogTable


def to_list(s):
    s = s.replace('\'', '')
    s = s.replace(' ', '')
    s = s.replace('[', '')
    s = s.replace(']', '')

    return s.split(',')


def map_tracking(rec):
    event_type = rec.get('EventType', '')
    if event_type != 'EVENT_AD_INSTALL':
        return

    attribution_method = rec.get('AttributionMethod')
    if attribution_method == 'fingerprint':
        # CRYPTR-164
        return

    uuid = rec.get('UUID')
    click_timestamp = rec.get('ClickTimestamp')

    if click_timestamp and uuid:
        url_keys = to_list(rec.get('UrlParameters_Keys', ''))

        if 'yandexuid' in url_keys:  # conflicts with YandexUidRu - 2 different yuids. 46k rows at 29.04.17
            i = url_keys.index('yandexuid')
            url_values = to_list(rec.get('UrlParameters_Values', ''))
            if i < len(url_values):
                yuid = url_values[i]
                if yuid:
                    ts = int(click_timestamp)
                    yield {config.ID_TYPE_UUID: uuid, config.ID_TYPE_YUID: yuid,
                           'ts': -ts, 'source': config.ID_SOURCE_TYPE_TRACK}
                    yield SoupDailyLogTable.make_rec(
                        yuid,
                        uuid,
                        soup_config.yuid_uuid_mobtrack,
                        ts=-ts,
                        table_index=1
                    )
        yuid = rec.get('YandexUidRu')
        if yuid:
            ts = int(click_timestamp)
            yield {config.ID_TYPE_UUID: uuid, config.ID_TYPE_YUID: yuid,
                   'ts': -ts, 'source': config.ID_SOURCE_TYPE_TRACK}
            yield SoupDailyLogTable.make_rec(
                yuid,
                uuid,
                soup_config.yuid_uuid_mobtrack,
                ts=-ts,
                table_index=1
            )


def import_tracking_log(tracking_logs, dt, workdir, devid_raw_f, soup_log):
    mr.mkdir(workdir)
    mr.mkdir(devid_raw_f)
    soup_log.ensure_dir()

    log_t = workdir + '%s_%s_%s_tmp' % (config.ID_TYPE_UUID, config.ID_TYPE_YUID,
                                        config.ID_SOURCE_TYPE_TRACK)
    out_t = devid_raw_f + '%s_%s_%s' % (config.ID_TYPE_UUID, config.ID_TYPE_YUID,
                                        config.ID_SOURCE_TYPE_TRACK)

    yt.run_map(
        map_tracking,
        tracking_logs,
        [
            log_t,
            soup_log.create(),
        ]
    )

    soup_log.prepare_daily_tables_from_log()

    yt.run_sort(log_t, sort_by=[config.ID_TYPE_UUID, config.ID_TYPE_YUID, 'ts'])

    yt.run_reduce(partial(reduce_device_log_events_day, dt=dt,
                          source_type=config.ID_SOURCE_TYPE_TRACK),
                  log_t, out_t,
                  sort_by=[config.ID_TYPE_UUID, config.ID_TYPE_YUID, 'ts'],
                  reduce_by=[config.ID_TYPE_UUID, config.ID_TYPE_YUID])

    finalize_device_yuid_day_tables([out_t])


class ImportMobileTrackingLogsDayTask(base_luigi_task.BaseTask):
    date = luigi.Parameter()
    run_date = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(ImportMobileTrackingLogsDayTask, self).__init__(*args, **kwargs)
        self.soup_log = SoupDailyLogTable(soup_config.LOG_SOURCE_MOBILE_TRACKING, self.date)

    def requires(self):
        return [yt_luigi.ExternalInput(config.LOG_FOLDERS['mob_tracking'] + self.date),
                yt_luigi.ExternalInput(config.LOG_FOLDERS['mob_tracking_private'] + self.date)]

    def run(self):
        out_indevice_folder = config.INDEVICE_YT_FOLDER + self.date + '/'
        out_date_folder = out_indevice_folder + 'perfect/'
        mr.mkdir(out_date_folder)

        tracking_logs = [r.table for r in self.requires()]

        import_tracking_log(tracking_logs,
                            self.date, out_date_folder + 'logs/',
                            out_date_folder + 'devid_raw_day/',
                            self.soup_log)

    def output(self):
        if self.date == self.run_date:
            soup_out_tables = self.soup_log.daily_tables_targets()
        else:
            soup_out_tables = []

        out_indevice_folder = config.INDEVICE_YT_FOLDER + self.date + '/'
        out_date_folder = out_indevice_folder + 'perfect/devid_raw_day/'
        return [yt_luigi.YtTarget(out_date_folder + '%s_%s_%s' % (config.ID_TYPE_UUID, config.ID_TYPE_YUID,
                                                                 config.ID_SOURCE_TYPE_TRACK))] + soup_out_tables


if __name__ == '__main__':
    yt.config.set_proxy(config.MR_SERVER)

    dt = sys.argv[1]

    if luigi.run(['ImportMobileTrackingLogsDayTask', '--date', dt, '--run-date', dt, '--workers', '10', '--local-scheduler']):
        sys.exit(0)
    else:
        sys.exit(1)
    pass
