import os
import re
import urlparse
from functools import partial

import luigi
import yt.wrapper as yt

from data_imports.day_aggregate import reduce_yuid_log_events_day, finalize_yuid_with_x_day_tables
from graph_barnavig import parse_uuid
from lib.luigi import yt_luigi
from rtcconf import config
from utils import mr_utils as mr
from utils import utils
from v2.soup import soup_config
from v2.soup.soup_tables import SoupDailyLogTable

PUNTO_BROWSERS = ['yu',
                  'yu_ch',
                  'yu_opch',
                  'yu_yb',
                  'yu_amg',
                  'yu_edge',
                  'yu_ff',
                  'yu_op',
                  'yu_op64',
                  'yu_ie']


def map_log(rec, dt):
    try:
        params = urlparse.parse_qs(urlparse.urlparse(rec['request'])[4])
        ui = params.get('ui', [''])[0]
        distr_yuid = params.get('distr_yandexuid', [''])[0]
        yasoft = params.get('yasoft', [''])[0]
        ext_bro_cookies = params.get('ckp', [''])[0].split('.')
        if ext_bro_cookies == ['']:
            ext_bro_cookies = []

        cookies = rec['cookies']
        yuid = mr.get_field_value('yandexuid', cookies, '; ')

        # TODO: use fuid and login to make pairs too
        # fuid = mr.get_field_value('fuid01', cookies, '; ')
        # login = mr.get_field_value('yandex_login', cookies, '; ')

        ui = parse_uuid(ui)
        if ui and yuid and re.match(r'\d+', yuid):
            yield {'id_value': ui, 'yuid': yuid.lstrip('0'), 'id_date': dt, '@table_index': 0}
            yield SoupDailyLogTable.make_rec(
                yuid.lstrip('0'),
                ui,
                soup_config.yuid_ui_ealeal,
                table_index=3
            )

        if ui and distr_yuid and re.match(r'\d+', distr_yuid):
            yield {'id_value': ui, 'yuid': distr_yuid.lstrip('0'), 'id_date': dt, '@table_index': 0}
            yield SoupDailyLogTable.make_rec(
                distr_yuid.lstrip('0'),
                ui,
                soup_config.yuid_ui_ealeal,
                table_index=3
            )

        if ui and yasoft in ('punto', 'puntomac'):
            for yu_source in PUNTO_BROWSERS:
                yuid = params.get(yu_source, [''])[0]
                if yuid:
                    yield {'id_value': ui, 'yuid': yuid, 'id_date': dt, '@table_index': 1}
                    yield SoupDailyLogTable.make_rec(
                        yuid,
                        ui,
                        soup_config.yuid_ui_puntoeal,
                        table_index=3
                    )

        if ui and ext_bro_cookies:
            for yuid in ext_bro_cookies:
                yield {'id_value': ui, 'yuid': yuid, 'id_date': dt, '@table_index': 2}
                yield SoupDailyLogTable.make_rec(
                    yuid,
                    ui,
                    soup_config.yuid_ui_extbrows,
                    table_index=3
                )
    except:
        pass

def run_eal(out_folder, log_table, dt, soup_log):

    soup_log.ensure_dir()

    out_eal_table = os.path.join(out_folder, 'yuid_with_' + config.ID_SOURCE_TYPE_EAL)
    out_eal_table_tmp = os.path.join(out_folder, 'yuid_with_' + config.ID_SOURCE_TYPE_EAL + '_tmp')
    out_punto_table = os.path.join(out_folder, 'yuid_with_' + config.ID_SOURCE_TYPE_PUNTO)
    out_punto_table_tmp = os.path.join(out_folder, 'yuid_with_' + config.ID_SOURCE_TYPE_PUNTO + '_tmp')
    out_ext_bro_table = os.path.join(out_folder, 'yuid_with_' + config.ID_SOURCE_TYPE_EXTERNAL_BROWSERS)
    out_ext_bro_table_tmp = os.path.join(out_folder, 'yuid_with_' + config.ID_SOURCE_TYPE_EXTERNAL_BROWSERS + '_tmp')
    
    yt.run_map(
        partial(map_log, dt=dt),
        log_table,
        [
            out_eal_table_tmp,
            out_punto_table_tmp,
            out_ext_bro_table_tmp,
            soup_log.create()
        ],
        spec=mr.DATA_SIZE_PER_JOB_2GB_SPEC
    )

    soup_log.prepare_daily_tables_from_log()

    mr.sort_all([out_eal_table_tmp, out_punto_table_tmp, out_ext_bro_table_tmp], sort_by='yuid')
    yt.run_reduce(partial(reduce_yuid_log_events_day, dt=dt,
                          id_type=config.ID_TYPE_BAR_UI,
                          source_type=config.ID_SOURCE_TYPE_EAL),
                  out_eal_table_tmp,
                  out_eal_table,
                  reduce_by='yuid')
    yt.run_reduce(partial(reduce_yuid_log_events_day, dt=dt,
                          id_type=config.ID_TYPE_BAR_UI,
                          source_type=config.ID_SOURCE_TYPE_PUNTO),
                  out_punto_table_tmp,
                  out_punto_table,
                  reduce_by='yuid')
    yt.run_reduce(partial(reduce_yuid_log_events_day, dt=dt,
                          id_type=config.ID_TYPE_BAR_UI,
                          source_type=config.ID_SOURCE_TYPE_EXTERNAL_BROWSERS),
                  out_ext_bro_table_tmp,
                  out_ext_bro_table,
                  reduce_by='yuid')

    finalize_yuid_with_x_day_tables([
        out_eal_table,
        out_punto_table,
        out_ext_bro_table,
    ])

    mr.drop(out_eal_table_tmp)
    mr.drop(out_punto_table_tmp)
    mr.drop(out_ext_bro_table_tmp)


class ImportEalDayTask(yt_luigi.BaseYtTask):
    date = luigi.Parameter()
    run_date = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(ImportEalDayTask, self).__init__(*args, **kwargs)
        self.soup_log = SoupDailyLogTable(soup_config.LOG_SOURCE_EAL, self.date)

    def input_folders(self):
        return {
            'statbox': config.STATBOX_EAL_FOLDER,
        }

    def output_folders(self):
        return {
            'raw': config.YT_OUTPUT_FOLDER + self.date + '/yuid_raw/',
        }

    def requires(self):
        return yt_luigi.ExternalInput(self.in_f('statbox') + self.date)

    def run(self):
        mr.mkdir(self.out_f('raw'))
        run_eal(self.out_f('raw'), self.in_f('statbox') + self.date, self.date, self.soup_log)

    def output(self):
        if self.date == self.run_date:
            soup_out_tables = self.soup_log.daily_tables_targets()
        else:
            soup_out_tables = []

        return [yt_luigi.YtTarget(self.out_f('raw') + 'yuid_with_' + config.ID_SOURCE_TYPE_EAL),
                yt_luigi.YtTarget(self.out_f('raw') + 'yuid_with_' + config.ID_SOURCE_TYPE_PUNTO),
                yt_luigi.YtTarget(self.out_f('raw') + 'yuid_with_' + config.ID_SOURCE_TYPE_EXTERNAL_BROWSERS)] + soup_out_tables


