import sys
from functools import partial

import luigi
import yt.wrapper as yt

from data_imports.day_aggregate import reduce_yuid_log_events_day, finalize_yuid_with_x_day_tables
from lib.luigi import yt_luigi
from rtcconf import config
from utils import mr_utils as mr
from utils import utils
from v2.soup import soup_config
from v2.soup.soup_tables import SoupDailyLogTable


def map_passport_sensitive_phone_md5(rec):
    out = {'id_type': config.ID_TYPE_PHONE,
           'source_type': config.ID_SOURCE_TYPE_PASSPORT_SENSITIVE,
           'key': rec['uid'],
           'puid': rec['uid']}
    if rec['event'] == 'account_modification' and rec['entity'] == 'phone':
        out['attribute'] = rec['attribute']
        out['entity_id'] = rec['entity_id']
        if rec['attribute'] == 'number' and rec['new']:
            phone = rec['new']
            id_prefix, id_hash = utils.prepare_phone_md5(phone)  # can't keep original phone from passport
            out['id_prefix'] = id_prefix
            out['id_value'] = id_hash
            yield SoupDailyLogTable.make_rec(
                rec['uid'],
                id_hash,
                soup_config.puid_phonehash_psphnbind,
                table_index=1
            )
        yield out


def reduce_bound_phones(key, recs):
    bounded = False
    id_prefix = None
    id_value = None
    for rec in recs:
        bounded = bounded or rec['attribute'] == 'bound'
        if 'id_prefix' in rec:
            id_prefix = rec['id_prefix']
            id_value = rec['id_value']
        if bounded and id_prefix:
            yield {'key': key['puid'],
                   'puid': key['puid'],
                   'id_prefix': id_prefix,
                   'id_value': id_value,
                   'id_type': rec['id_type'],
                   'source_type': rec['source_type']}
            break


class ImportPassportPhoneBindingsDayTask(yt_luigi.BaseYtTask):
    date = luigi.Parameter()
    run_date = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(ImportPassportPhoneBindingsDayTask, self).__init__(*args, **kwargs)
        self.soup_log = SoupDailyLogTable(soup_config.LOG_SOURCE_PASSPORT_PHONE_BINDINGS, self.date)

    def input_folders(self):
        return {
            'passport_sensitive_log': config.LOG_FOLDERS['passport_sensitive']
        }

    def output_folders(self):
        return {
            'yuid_raw_day': config.YT_OUTPUT_FOLDER + self.date + '/yuid_raw/',
            'passport_sensitive': config.YT_OUTPUT_FOLDER + self.date + '/passport_sensitive/'
        }

    def prepare_passport_log(self, passport_sensitive_log, out_passport_sens, out_yuid_raw, soup_log):
        dt = self.date
        mr.mkdir(out_passport_sens)
        mr.mkdir(out_yuid_raw)
        soup_log.ensure_dir()

        yt.run_map(
            map_passport_sensitive_phone_md5,
            passport_sensitive_log,
            [
                out_passport_sens + 'puid_md5phone_passport_day_raw',
                soup_log.create(),
            ]
        )
        soup_log.prepare_daily_tables_from_log()
    
        yt.run_sort(out_passport_sens + 'puid_md5phone_passport_day_raw', sort_by=['puid', 'entity_id'])
        yt.run_reduce(reduce_bound_phones,
                      out_passport_sens + 'puid_md5phone_passport_day_raw',
                      out_passport_sens + 'puid_md5phone_passport_day',
                      reduce_by=['puid', 'entity_id'])
        yt.run_sort(out_passport_sens + 'puid_md5phone_passport_day', sort_by='puid')

        out_table = out_yuid_raw + 'puid_with_' + config.ID_TYPE_PHONE + '_' + config.ID_SOURCE_TYPE_PASSPORT_SENSITIVE
        yt.run_reduce(partial(reduce_yuid_log_events_day, dt=dt,
                              id_type=config.ID_TYPE_PHONE,
                              source_type=config.ID_SOURCE_TYPE_PASSPORT_SENSITIVE,
                              key_col='puid'),
                      out_passport_sens + 'puid_md5phone_passport_day',
                      out_table,
                      reduce_by='puid')

        finalize_yuid_with_x_day_tables([out_table], key_col='puid')

    def requires(self):
        return yt_luigi.ExternalInput(self.in_f('passport_sensitive_log') + self.date)

    def run(self):
        passport_sensitive_log = config.LOG_FOLDERS['passport_sensitive'] + self.date
        self.prepare_passport_log(passport_sensitive_log,
                                  self.out_f('passport_sensitive'),
                                  self.out_f('yuid_raw_day'),
                                  self.soup_log)

    def output(self):
        if self.date == self.run_date:
            soup_out_tables = self.soup_log.daily_tables_targets()
        else:
            soup_out_tables = []

        t = self.out_f('yuid_raw_day') + 'puid_with_' + config.ID_TYPE_PHONE + '_' + config.ID_SOURCE_TYPE_PASSPORT_SENSITIVE
        return [yt_luigi.YtTarget(t)] + soup_out_tables


if __name__ == '__main__':
    yt.config.set_proxy(config.MR_SERVER)

    dt = sys.argv[1]
    luigi.run(['ImportPassportPhoneBindingsDayTask', '--date', dt, '--run-date', dt, '--workers', '10', '--local-scheduler'])
