from functools import partial

import luigi
import yt.wrapper as yt

from data_imports.day_aggregate import reduce_yuid_log_events_day
from lib.luigi import yt_luigi
from rtcconf import config
from utils import mr_utils as mr
from v2.soup import soup_config
from v2.soup.soup_tables import SoupDailyLogTable


def map_log(rec, dt):
    email = rec.get('email', '')
    yuid = rec.get('yandexuid', '')
    if yuid and email and email != 'None' and '@' in email:
        yield {'id_value': email, 'yuid': yuid, 'id_date': dt}
        yield SoupDailyLogTable.make_rec(
            yuid,
            email,
            soup_config.yuid_email_senderday,
            table_index=1
        )


class ImportSenderDayTask(yt_luigi.BaseYtTask):

    date = luigi.Parameter()
    run_date = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(ImportSenderDayTask, self).__init__(*args, **kwargs)
        self.soup_log = SoupDailyLogTable(soup_config.LOG_SOURCE_SENDERDAY, self.date)

    def input_folders(self):
        return {
            'statbox': config.STATBOX_SENDER_FOLDER
        }

    def output_folders(self):
        return {
            'graph': config.YT_OUTPUT_FOLDER + self.date + '/'
        }

    def requires(self):
        return yt_luigi.ExternalInput(self.in_f('statbox') + self.date)

    def run(self):
        mr.mkdir(self.out_f('graph') + 'yuid_raw')
        self.soup_log.ensure_dir()

        temporary_table = self.out_f('graph') + 'yuid_raw/temporary_table_tickets'

        yt.run_map(
            partial(
                map_log,
                dt=self.date
            ),
            self.in_f('statbox') + self.date,
            [
                temporary_table,
                self.soup_log.create()
            ]
        )

        self.soup_log.prepare_daily_tables_from_log()

        yt.run_map_reduce(
            None,
            partial(
                reduce_yuid_log_events_day,
                dt=self.date,
                id_type=config.ID_TYPE_EMAIL,
                source_type=config.ID_SOURCE_TYPE_SENDER
            ),
            temporary_table,
            self.out_f('graph') + 'yuid_raw/yuid_with_' + config.ID_SOURCE_TYPE_SENDER,
            sort_by='yuid',
            reduce_by='yuid'
        )

        mr.drop(temporary_table)

    def output(self):

        if self.date == self.run_date:
            soup_out_tables = self.soup_log.daily_tables_targets()
        else:
            soup_out_tables = []

        return [yt_luigi.YtTarget(self.out_f('graph') + 'yuid_raw/yuid_with_' + config.ID_SOURCE_TYPE_SENDER,
                                  allow_empty=True)] + soup_out_tables
