import json
from functools import partial

import luigi
import yt.wrapper as yt

from data_imports.day_aggregate import reduce_yuid_log_events_day, reduce_device_log_events_day, \
    finalize_yuid_with_x_day_tables, finalize_device_yuid_day_tables
from lib.luigi import base_luigi_task
from lib.luigi import yt_luigi
from rtcconf import config
from utils import mr_utils as mr
from utils import utils
from v2.soup import soup_config
from v2.soup.soup_tables import SoupDailyLogTable

OOM_LIMIT = 100


def map_lookup_access_log(rec):
    request = rec.get('request')
    if request and 'yasoft=' in request:
        uuid = mr.get_field_value('uuid', request, '&')
        yasoft = mr.get_field_value('yasoft', request, '&')

        yuid_old = mr.get_field_value('yandexuid', request, '&')
        yuid_ru = mr.get_field_value('yandexuid.ru', request, '&')
        yuid = yuid_ru if yuid_ru else yuid_old

        try:
            ts = utils.get_ts(rec.get('iso_eventtime', ''), rec.get('timezone', ''))
        except ValueError:
            return

        if uuid and yuid:
            yield {'uuid': uuid, 'yuid': yuid, 'ts': -ts,
                   'source': config.ID_SOURCE_TYPE_YABROWSER_IOS, 'yasoft': yasoft}
            yield SoupDailyLogTable.make_rec(
                yuid,
                uuid,
                soup_config.yuid_uuid_sbapial,
                ts=-ts,
                table_index=1
            )


def run_day(log_table, workdir, devid_raw_folder, dt, soup_log):
    mr.mkdir(workdir)
    mr.mkdir(devid_raw_folder)
    soup_log.ensure_dir()

    yt.run_map(map_lookup_access_log,
               log_table,
               [
                   workdir + 'uuid_yuid_' + config.ID_SOURCE_TYPE_YABROWSER_IOS,
                   soup_log.create()
               ],
               spec=mr.DATA_SIZE_PER_JOB_2GB_SPEC)

    soup_log.prepare_daily_tables_from_log()

    mr.merge(workdir + 'uuid_yuid_' + config.ID_SOURCE_TYPE_YABROWSER_IOS)

    yt.run_sort(workdir + 'uuid_yuid_' + config.ID_SOURCE_TYPE_YABROWSER_IOS,
                sort_by=[config.ID_TYPE_UUID, config.ID_TYPE_YUID, 'ts'])

    uuid_yuid_yabro_ios = devid_raw_folder + 'uuid_yuid_' + config.ID_SOURCE_TYPE_YABROWSER_IOS
    yt.run_reduce(partial(reduce_device_log_events_day,
                          dt=dt, source_type=config.ID_SOURCE_TYPE_YABROWSER_IOS),
                  workdir + 'uuid_yuid_' + config.ID_SOURCE_TYPE_YABROWSER_IOS,
                  uuid_yuid_yabro_ios,
                  sort_by=[config.ID_TYPE_UUID, config.ID_TYPE_YUID, 'ts'],
                  reduce_by=[config.ID_TYPE_UUID, config.ID_TYPE_YUID])

    finalize_device_yuid_day_tables([uuid_yuid_yabro_ios])


def map_mitb(rec):
    data = rec.get('data')
    if data and data != '-':
        try:
            json_data = json.loads(data.decode('string_escape'))
            ui = json_data['ui'].strip("{}")
            yuids_bro = dict()
            for yuid_json in json_data['yandexuids']:
                yuid = yuid_json['value']
                if yuid:
                    yuids_bro[yuid] = yuid_json['browser']

            for yuid, browser in yuids_bro.iteritems():
                yield {'id_value': ui, 'yuid': yuid, 'browser': browser}
                yield SoupDailyLogTable.make_rec(
                    yuid,
                    ui,
                    soup_config.yuid_ui_sbapimitblog,
                    table_index=2
                )

        except Exception:
            yield {'data': data, '@table_index': 1}


class ImportSbApiAccessLogDayTask(base_luigi_task.BaseTask):

    date = luigi.Parameter()
    run_date = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(ImportSbApiAccessLogDayTask, self).__init__(*args, **kwargs)
        self.soup_log = SoupDailyLogTable(soup_config.LOG_SOURCE_SBAPIACCESSLOG, self.date)

    def requires(self):
        return yt_luigi.ExternalInput(config.STATBOX_SBAPI_LOOKUP_LOG_FOLDER + self.date)

    def run(self):
        log_table = config.STATBOX_SBAPI_LOOKUP_LOG_FOLDER + self.date
        workdir = config.YT_OUTPUT_FOLDER + self.date + '/sbapi/'
        devid_raw_folder = config.INDEVICE_YT_FOLDER + self.date + '/perfect/devid_raw_day/'

        run_day(log_table, workdir, devid_raw_folder, self.date, self.soup_log)


    def output(self):
        devid_raw_folder = config.INDEVICE_YT_FOLDER + self.date + '/perfect/devid_raw_day/'
        return yt_luigi.YtTarget(devid_raw_folder + 'uuid_yuid_' + config.ID_SOURCE_TYPE_YABROWSER_IOS)


class ImportSbApiMitbLogDayTask(yt_luigi.BaseYtTask):

    date = luigi.Parameter()
    run_date = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(ImportSbApiMitbLogDayTask, self).__init__(*args, **kwargs)
        self.soup_log = SoupDailyLogTable(soup_config.LOG_SOURCE_SBAPIMITBLOG, self.date)

    def input_folders(self):
        return {
            'log': config.STATBOX_SBAPI_MITB_LOG_FOLDER
        }

    def output_folders(self):
        return {
            'sbapi_tmp': config.YT_OUTPUT_FOLDER + self.date + '/sbapi/',
            'yuid_raw_day': config.YT_OUTPUT_FOLDER + self.date + '/yuid_raw/'
        }

    def requires(self):
        return yt_luigi.ExternalInput(self.in_f('log') + self.date)

    def run(self):
        mr.mkdir(self.out_f('sbapi_tmp'))
        mr.mkdir(self.out_f('yuid_raw_day'))
        self.soup_log.ensure_dir()

        yt.run_map(
            map_mitb,
            self.in_f('log') + self.date,
            [
                self.out_f('sbapi_tmp') + 'yuid_ui_mitb_log',
                self.out_f('sbapi_tmp') + 'yuid_ui_mitb_log_errrs',
                self.soup_log.create(),
            ]
        )

        self.soup_log.prepare_daily_tables_from_log()

        out_table = self.out_f(
            'yuid_raw_day') + 'yuid_with_' + config.ID_TYPE_BAR_UI + '_' + config.ID_SOURCE_TYPE_BROWSER_MANAGER
        yt.run_map_reduce(None, partial(reduce_yuid_log_events_day, dt=self.date,
                                        id_type=config.ID_TYPE_BAR_UI,
                                        source_type=config.ID_SOURCE_TYPE_BROWSER_MANAGER),
                          self.out_f('sbapi_tmp') + 'yuid_ui_mitb_log',
                          out_table,
                          reduce_by=config.ID_TYPE_YUID)

        finalize_yuid_with_x_day_tables([out_table])


    def output(self):
        if self.date == self.run_date:
            soup_out_tables = self.soup_log.daily_tables_targets()
        else:
            soup_out_tables = []

        return [yt_luigi.YtTarget(self.out_f('yuid_raw_day') +
                                 'yuid_with_' + config.ID_TYPE_BAR_UI + '_' + config.ID_SOURCE_TYPE_BROWSER_MANAGER)] + soup_out_tables


if __name__ == '__main__':
    yt.config.set_proxy(config.MR_SERVER)
    yt.config["tabular_data_format"] = yt.YsonFormat(process_table_index=True)

    # log_table = config.STATBOX_SBAPI_LOOKUP_LOG_FOLDER + dt
    workdir = '//home/crypta/team/artembelov/mitb/'
    mr.mkdir(workdir)
    #
    # yt.run_map(map_lookup_access_log, log_table, workdir + 'uuid_yuid_ios_yabrowser')
    # yt.run_sort(workdir + 'uuid_yuid_ios_yabrowser', sort_by='uuid')

    # log_tables = mr.get_date_tables('//home/logfeller/logs/sbapi-access-mitb-log/1d/', None, 30)
    # yt.run_map(map_mitb, log_tables, [workdir + 'log', workdir + 'wat'])
    # #
    # mr.distinct_by(['id_value', 'yuid', 'browser'], workdir + 'log', workdir + 'yuid_ui')
    #
    # yt.run_sort(workdir + 'yuid_ui', sort_by='yuid')
    #
    # yt.run_reduce(mr.filter_left_by_right, [
    #     config.GRAPH_YT_DICTS_FOLDER + 'yuid_with_all',
    #     workdir + 'yuid_ui'
    # ], workdir + 'yuid_found',
    #               reduce_by='yuid')

