# coding=utf-8
import luigi
from textwrap import dedent

from lib.luigi import yt_luigi
from rtcconf import config
from v2.soup import soup_config
from v2.soup.soup_tables import SoupDailyLogTable

from utils.yql_utils import run_yql


YQL = dedent('''
    $pair_types = AsList(
        {edge_types}
    );

    $source_type = '{source_type}';
    $log_source = '{log_source}';

    $tuplize_ids = ($x) -> {{ return ($x.Type, $x.Value) }};

    $make_pairs = ($pair, $dict) -> {{
        $left_type = $pair.0;
        $right_type = $pair.1;
        $left = $dict{{$left_type}};
        $right = $dict{{$right_type}};
        return if (
            $left is not null and $right is not null,
            ($left, $left_type, $right, $right_type),
            null
        );
    }};

    $not_null = ($x) -> {{ return $x is not null }};

    $parsed_log = (
        select * from (
            select CryptaIdserv::ParseIdentifyRequest(query) as query, -cast(unixtime as int64) as ts
            from [{input_table}]
        ) where query.OnlineMatching
    );

    $id_dicts = (
        select
            ToDict(ListMap(query.Ids ?? ListCreate(TypeOf(query.Ids)), $tuplize_ids)) as id_dict,
            ts
        from $parsed_log
    );

    $id_pair_lists = (
        select
            ListFilter(ListMap($pair_types, $make_pairs, id_dict), $not_null) as pair_list,
            ts
        from $id_dicts
    );

    insert into [{output_table}] with truncate
    select
        pair_list.0 as id1,
        pair_list.1 as id1Type,
        pair_list.2 as id2,
        pair_list.3 as id2Type,
        $source_type as sourceType,
        $log_source as logSource,
        ts as ts,
        null as hits
    from $id_pair_lists
    flatten by pair_list;
''')


def mk_yql_edge_types(edge_types):
    return ',\n'.join(["('{}', '{}')".format(x.id1_type, x.id2_type) for x in edge_types])


class ImportIdservAccessLog(luigi.Task):
    date = luigi.Parameter()
    run_date = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(ImportIdservAccessLog, self).__init__(*args, **kwargs)
        self.soup_log = SoupDailyLogTable(soup_config.LOG_SOURCE_IDSERV, self.date)

    def requires(self):
        return yt_luigi.ExternalInput(self.get_log_table())

    def output(self):
        return self.soup_log.daily_tables_targets()

    def run(self):
        self.soup_log.ensure_dir()
        soup_tmp_table = self.soup_log.create()

        query = YQL.format(
            edge_types=mk_yql_edge_types(soup_config.idserv_edge_types),
            input_table=self.get_log_table(),
            output_table=soup_tmp_table,
            source_type=soup_config.ID_SOURCE_TYPE_IDSERV,
            log_source=soup_config.LOG_SOURCE_IDSERV
        )
        run_yql(query=query)

        self.soup_log.prepare_daily_tables_from_log()

    def get_log_table(self):
        return config.LOG_FOLDERS['idserv_access'] + self.date
