import logging
import sys
import os

import luigi
import yt.wrapper as yt

from data_imports.import_logs.webvisor import webvisor_stats
from infra import graph_history_snapshot
from infra import graph_postproc
from lib import yt_trace
from lib.crypta_api.crypta_task_status_api import CryptaTaskStatusApi
from lib.luigi import base_luigi_task
from lib.luigi import graph_fail_by_design
from lib.luigi import yt_luigi
from lib.luigi.yt_luigi import PostGraphTask
from lib.sandbox.env import get_appropriate_sandbox_env
from matching import graph_resulted_pairs_tables
from matching.common_stats import fpc_matching_stat
from matching.common_stats import graph_macs_count_stat
from matching.common_stats import graph_stat_report
from matching.device_matching.app_metrica import app_metrica_month
from matching.device_matching.distr import graph_cube_metrica
from matching.device_matching.distr import yuid_distr_ui_desktop
from matching.device_matching.distr.device_yuid_distr_dicts import UpdateYuidDevidIndeviceAllDict
from matching.device_matching.stats import graph_new_indevice_pairs_metrica
from matching.device_matching.stats import graph_stat_apps
from matching.export import graph_upload_bb
from matching.export import graph_upload_is
from matching.human_matching import graph_vertices
from matching.human_matching.graph_clustering import ClusterVertices, ClusteringConfig
from matching.human_matching.graph_vertices_base import VerticesConfig
from matching.human_matching.stats import graph_stat_vertices
from matching.human_matching.stats import toloka_metrics_indevice
from matching.human_matching.stats.radius import radius_metrics_indevice
from matching.human_matching.upload import bb_hubs
from matching.pairs.stats import graph_pairs_sources_stat
from matching.pairs.stats import graph_stat_pairs
from matching.pairs.stats import graph_stat_passport
from matching.pairs import graph_pairs_black_list
from matching.yuid_matching import graph_dict
from matching.yuid_matching.stats import graph_precision_metrics
from matching.yuid_matching.stats import graph_stat_id
from rtcconf import config
from utils import logging_setup
from utils import mr_utils as mr
from utils import utils
from v2 import sandbox_tasks, yuid_apps
from v2.ids_storage import dicts_to_storage
from v2.soup import graph_soup, soup_config
from household import hh_to_bb


class GraphAllTask(luigi.WrapperTask):
    date = luigi.Parameter()

    def requires(self):
        # exact
        exact_vertices_task = graph_vertices.GraphVerticesExact(self.date,
                                                                vertices_type='exact',
                                                                yuid_pairs_folder='pairs/')
        exact_cluster = ClusterVertices(exact_vertices_task.vertices_config, ClusteringConfig())

        # fuzzy
        fuzzy_vertices_task = graph_vertices.GraphVerticesFuzzy(self.date)
        # TODO: local clustering for fuzzy was turned off due to yt read performance
        fuzzy_cluster = ClusterVertices(fuzzy_vertices_task.vertices_config,
                                        ClusteringConfig(max_size_component=20,
                                                         local_clustering_enabled=False))

        # clustering experiment
        experiment_clustering_config = ClusteringConfig(new_multi_source_approach=True)
        exact_cluster_experiment = ClusterVertices(exact_vertices_task.vertices_config,
                                                   clustering_config=experiment_clustering_config,
                                                   name='cluster_experiment')

        # no-login vertices experiment
        # no_login_experiment = ClusterVertices(graph_vertices.GraphVerticesNoLogin(self.date).vertices_config,
        #                                       ClusteringConfig())

        all_vertices_tasks = [
            fuzzy_vertices_task,
            exact_vertices_task,
            exact_cluster,
            fuzzy_cluster,
            exact_cluster_experiment,
            # no_login_experiment
        ]

        main_vertices_config = None
        to_bb = []

        for task in all_vertices_tasks:
            vertices_config = task.vertices_config
            if vertices_config.vertices_type in config.VERTICES_EXPERIMENTS:
                vertices_config.bb_experiment = True
                to_bb.append(vertices_config)

                if not config.VERTICES_EXPERIMENTS[vertices_config.vertices_type]:  # no experiment_id means production
                    if main_vertices_config:
                        raise Exception('You can\'t use two types of vertices in prod. ' +
                                        'Conflict %s vs %s' % (
                                         main_vertices_config.vertices_type, vertices_config.vertices_type))
                    vertices_config.main_vertices = True
                    main_vertices_config = vertices_config

        if not main_vertices_config:
            raise Exception('At least one production vertices should be set')

        vertices_to_dict = graph_vertices.CopyVerticesToDict(self.date,
                                                             exact_task=exact_cluster,
                                                             fuzzy_task=fuzzy_cluster)

        tasks = [
            # vertices related stats
            graph_stat_vertices.VerticesAllStatsTask(
                vertices_configs=[task.vertices_config for task in all_vertices_tasks]
            ),

            graph_stat_report.MatchingCoverageReportTask(date=self.date, name='statface-done',
                                                         vertices_config=fuzzy_vertices_task.vertices_config,
                                                         legacy_report=True),
            graph_stat_report.MatchingCoverageReportTask(date=self.date, name='statface-exact-done',
                                                         vertices_config=exact_cluster.vertices_config),
            graph_stat_report.MatchingCoverageReportTask(date=self.date, name='statface-fuzzy-done',
                                                         vertices_config=fuzzy_cluster.vertices_config),

            radius_metrics_indevice.RadiusMetricsIndevice(date=self.date,
                                                          vertices_config=exact_vertices_task.vertices_config),

            # other stats
            graph_stat_passport.LoginShortSessionPairsStat(self.date),
            webvisor_stats.WebvisorDomainsStats(self.date),
            webvisor_stats.WebvisorDatesStats(self.date),
            graph_stat_apps.AppStatsToGraphite(date=self.date, name='app-stats-done'),
            graph_stat_pairs.PairsToGraphiteTask(date=self.date, name='graphite-done'),
            graph_stat_id.IdStatTask(date=self.date, name='id_stats'),
            graph_stat_id.YuidStatTask(self.date),
            graph_pairs_sources_stat.YuidIdBySourceStatsTask(date=self.date),

            graph_stat_report.GeoReportTask(date=self.date, name='geo-statface-done'),

            # uploads
            graph_upload_bb.UploadVerticesBBTask(date=self.date,
                                                 vertices_configs=to_bb),
            # graph_upload_bb.UploadMobileBBTask(self.date),
            graph_upload_is.UploadISTask(self.date,
                                         vertices_config=main_vertices_config),
            hh_to_bb.UploadHouseholdsBBTask(date=self.date),

            # for distribution
            yuid_distr_ui_desktop.UiYuidAllTask(self.date),
            app_metrica_month.UpdateUuidDevidIndeviceAllDict(self.date),
            UpdateYuidDevidIndeviceAllDict(self.date),

            # for external consumers
            vertices_to_dict,
            graph_dict.YuidWithIdXHash(self.date),

            # mac statistics
            graph_macs_count_stat.MacAddressesWithIdsStatTask(self.date, exact_task=exact_cluster, fuzzy_task=fuzzy_cluster),

            # create email,phone,puid with cryptaid,yuid,devid tables (after matching)
            graph_resulted_pairs_tables.CreateImportantPairsTablesAfterMatching(self.date, exact_task=exact_cluster, fuzzy_task=fuzzy_cluster),

            # precision of yuid-email and yuid-phone edges by source
            graph_precision_metrics.PrecisionYuidsStatisticsMainTask(self.date),

            # stat for realtime matching in sandbox (fpc-yuid mathcing)
            fpc_matching_stat.FpcStatTask(self.date),

            # metrica_sockets statistics
            graph_new_indevice_pairs_metrica.NewIndevicePairsStat(date=self.date),

            # v2
            dicts_to_storage.IdsStorageIsReady(date=self.date),
            graph_soup.SoupIsReady(date=self.date),
            #graph_supometriya.GraphSupometriyaTask(date=self.date)

            # black list
            graph_pairs_black_list.GraphPairsBlackList(date=self.date),

            # Mobile apps for mobile yuids via puid & indevice links
            yuid_apps.PrepareUploadYuidAppsTask(date=self.date)
        ]

        # yes toloka in tests too
        tasks += [
            toloka_metrics_indevice.TolokaMetrics(
                self.date,
                vertices_config=exact_vertices_task.vertices_config),

            toloka_metrics_indevice.TolokaMetrics(
                self.date,
                vertices_config=fuzzy_vertices_task.vertices_config),

            toloka_metrics_indevice.TolokaMetricsDevidCoverage(
                self.date,
                vertices_config=exact_vertices_task.vertices_config),

            toloka_metrics_indevice.TolokaMetricsHousehold(
                self.date,
                vertices_table=exact_vertices_task.vertices_config.get_vertices_table(),
                vertices_table_yuid_column='key',
                vertices_table_cryptaid_column='crypta_id',
                vertices_producing_task=exact_vertices_task,
                vertices_type=exact_vertices_task.vertices_config.vertices_type),

            toloka_metrics_indevice.TolokaMetricsHousehold(
                self.date,
                vertices_table=os.path.join(config.GRAPH_YT_OUTPUT_FOLDER, 'v2/matching/vertices_no_multi_profile'),
                vertices_table_yuid_column='id',
                vertices_table_cryptaid_column='cryptaId',
                vertices_producing_task=None,
                vertices_type='2_0'),
        ]

        if config.CRYPTA_ENV != 'development':
            tasks += [
                # no sandbox instance in tests
                # sandbox_tasks.PrepareSoupForMatching(self.date),  # being triggered by STEP now
                yuid_apps.UploadYuidAppsTask(self.date)
            ]

        # long term stability
        history_vertices_config = VerticesConfig("", "history", self.date,
                                                 base_path=config.YT_OUTPUT_FOLDER + 'history/')
        history_vertices_config_month_ago = history_vertices_config.get_previous_vertices_config(30)

        if history_vertices_config_month_ago and yt.exists(history_vertices_config_month_ago.get_vertices_table()):
            tasks.append(graph_stat_vertices.VerticesStabilityStats(
                history_vertices_config_month_ago,
                main_vertices_config,
                'stability30')
            )

        if config.HAS_BSYETI_TABLES == 'yes':
            tasks += [bb_hubs.CheckVerticesInBb(main_vertices_config)]

        if config.TEST_FAIL_REPORTING == 'yes':
            tasks += [
                graph_fail_by_design.FailingTask(self.date),
                graph_fail_by_design.FailingYTTask(self.date),
                graph_fail_by_design.FailingInsideYTTask(self.date)
            ]

        tasks.append(graph_cube_metrica.GraphCubeStatTask(self.date))

        return tasks

    def run(self):
        utils.monrun_ok()

class GraphPostprocTask(PostGraphTask):

    def __init__(self, date, name):
        super(GraphPostprocTask, self).__init__(date=date, name=name)

    def run_post_graph(self):
        graph_postproc.run_postproc_and_merge()

    def requires(self):
        return GraphAllTask(date=self.date)


class MainTask(luigi.WrapperTask):
    date = luigi.Parameter()

    def requires(self):
        graph_all_task = GraphAllTask(date=self.date)
        tasks = [graph_all_task,
                 GraphPostprocTask(date=self.date, name='postproc-done')]
        if config.CRYPTA_ENV == 'production':
            tasks.append(graph_history_snapshot.GraphHistorySnapshot(date=self.date, parent_task=graph_all_task))
        return tasks


crypta_task_status_api = CryptaTaskStatusApi(
    oauth_token="",  # no auth for now
    environment=get_appropriate_sandbox_env()
)

def report_task_status_to_api(task, status):
    if config.CRYPTA_ENV != 'development':
        crypta_task_status_api.report_task_status(
            task.get_task_family(),
            status,
            [],
            task.param_kwargs
        )

@luigi.Task.event_handler(luigi.Event.FAILURE)
def celebrate_failure(task, e):
    utils.monrun_luigi_error(task, e)
    report_task_status_to_api(task, "FAILURE")

@luigi.Task.event_handler(luigi.Event.PROCESS_FAILURE)
def on_process_failure(task, e):
    utils.monrun_luigi_error(task, e)
    report_task_status_to_api(task, "FAILURE")


@luigi.Task.event_handler(luigi.Event.DEPENDENCY_MISSING)
def on_missing_dependency(task):
    logging.warn("Missing dependency during execution: %s" % task)


@luigi.Task.event_handler(luigi.Event.SUCCESS)
def mourn_success(task):
    utils.monrun_luigi_ok(task)
    report_task_status_to_api(task, "SUCCESS")


@luigi.Task.event_handler(luigi.Event.START)
def on_task_start(task):
    yt_luigi.reset_global_yt_state()


if __name__ == '__main__':
    yt.config.set_proxy(config.MR_SERVER)
    yt.config["tabular_data_format"] = yt.YsonFormat(process_table_index=True)
    yt.config.INTERMEDIATE_DATA_ACCOUNT = 'crypta'

    logging_setup.setup_root_logger()
    logging_setup.setup_luigi_logging()
    logging_setup.setup_yt_logging()

    dt = sys.argv[1]

    mr.mkdir(config.GRAPH_YT_DICTS_FOLDER)
    mr.mkdir(config.YT_OUTPUT_FOLDER + dt + '/mobile/')
    mr.mkdir(config.YT_OUTPUT_FOLDER + dt + '/raw_links/')
    mr.mkdir(config.INDEVICE_YT_FOLDER + dt + '/perfect/')

    mr.mkdir(soup_config.SOUP_DIR)
    mr.mkdir(soup_config.SOUP_DAY_DIR)
    mr.mkdir(soup_config.SOUP_DUMPS_DIR)
    mr.mkdir(soup_config.SOUP_DAY_LOGS_DIR)
    mr.mkdir(soup_config.SOUP_DAY_LOGS_DIR + 'processing_status')

    yt_trace.setup_trace()

    retry_attempt = 0
    while retry_attempt < 3:
        if retry_attempt == 0:
            logging.info('Running luigi MainTask...')
        else:
            logging.info('Luigi failed. MainTask %d retry...' % retry_attempt)

        success = luigi.build([MainTask(dt)], workers=10, scheduler_port=int(config.LUIGID_PORT))
        if success:
            utils.monrun_ok()
            sys.exit(0)
        else:
            retry_attempt += 1

    utils.monrun_error('Luigi finally failed after 3 attempts: ')
    sys.exit(1)

