import base64
import math


def get_field_value(name, tskv):
    for item in tskv.split('\t'):
        pair = item.split('=', 1)
        if len(pair) == 2 and pair[0] == name:
            return pair[1]
    return ''


class HouseholdInfo():
    # order_idx, name, number_of_bits
    attr_meta = [(1, 'version', 2), (2, 'male', 1), (3, 'female', 1), (4, 'child', 1), (5, 'grand', 1), (6, 'inc', 2),
                 (7, 'cbrsrs', 4), (8, 'cindiv', 4), (9, 'ccoll', 4), (10, 'cstat', 4), (11, 'ctrav', 4),
                 (12, 'desk', 1), (13, 'mobapp', 1), (14, 'mobweb', 1), (15, 'smart', 1), (16, 'otherp', 1),
                 (17, 'win', 1), (18, 'ios', 1), (19, 'android', 1), (20, 'macos', 1), (21, 'otheros', 1),
                 # hh geo
                 (22, 'lat_sign', 1), (23, 'lat', 25), (24, 'lon_sign', 1), (25, 'lon', 25),
                 # main profile attributes (see CRYPTAIS-400)
                 (26, 'mp_male', 1), (27, 'mp_female', 1),
                 (28, 'mp_l18', 1), (29, 'mp_18_24', 1), (30, 'mp_25_34', 1), (31, 'mp_35_44', 1),
                 (32, 'mp_45_54', 1), (33, 'mp_g55', 1), (34, 'mp_align_bit_no_meaning', 1),
                 # adhocs
                 (35, 'cadhocs', 12), (36, 'c3dpartydata', 12),
                 (37, 'adhocs', -1), (38, '3dpartydata', -1)]

    def __init__(self, hh_data):
        self.id = hh_data.get('hh_id', 0)

        self.attrs = {}
        for attr in self.attr_meta:
            self.attrs[attr[1]] = 0

        self.attrs['version'] = 0
        self.genders = hh_data.get('sex', [])
        # [[0 - male], [1 - female]]
        self.attrs['male'] = 1 if '0' in self.genders else 0
        self.attrs['female'] = 1 if '1' in self.genders else 0

        self.ages = set(hh_data.get('age', []))
        self.attrs['child'] = 1 if '0' in self.ages else 0
        self.attrs['grand'] = 1 if '5' in self.ages else 0

        self.inc = hh_data.get('inc', '')
        self.attrs['inc'] = int(self.inc) + 1 if self.inc else 0

        self.brsrs = set(hh_data.get('brs', []))
        self.attrs['cbrsrs'] = min(len(self.brsrs), 15)

        self.yc = hh_data['yc']
        self.dc = hh_data['dc']
        self.oss = set(os.lower() for os in hh_data['oss'])
        self.pltfrms = hh_data['pltfrms']

        # fast fix; TODO: change hh bb format to support adhoc-ids with more than 12 bit length
        self.adhocs = set(a for a in hh_data['adhocs'] if int(a) < (1 << 12))

        self.thirdpty = set()

        if 'd' in self.pltfrms:
            self.attrs['desk'] = 1
        if self.dc > 0:
            self.attrs['mobapp'] = 1
        if 'm' in self.pltfrms:
            self.attrs['mobweb'] = 1

        if 'windows' in self.oss:
            self.attrs['win'] = 1
        if 'ios' in self.oss:
            self.attrs['ios'] = 1
        if 'android' in self.oss:
            self.attrs['android'] = 1
        if 'macos' in self.oss:
            self.attrs['macos'] = 1
        if len(self.oss - {'windows', 'ios', 'android', 'macos'}) > 0:
            self.attrs['otheros'] = 1

        self.attrs['adhocs'] = self.adhocs
        self.attrs['cadhocs'] = len(self.adhocs)

        self.attrs['3dpartydata'] = self.thirdpty
        self.attrs['c3dpartydata'] = len(self.thirdpty)

        # geo
        lat = float(hh_data.get('lat', 0))
        self.attrs['lat_sign'] = 0 if lat >= 0 else 1
        self.attrs['lat'] = int(math.fabs(lat)*(1<<17))

        lon = float(hh_data.get('lon', 0))
        self.attrs['lon_sign'] = 0 if lon >= 0 else 1
        self.attrs['lon'] = int(math.fabs(lon)*(1<<17))

        # main profile socdem flags
        mp_sex = hh_data['mp_sex']
        mp_age = hh_data['mp_age']
        mp_sex_bits = [(1 if str(i) in mp_sex else 0) for i in xrange(2)]
        mp_age_bits = [(1 if str(i) in mp_age else 0) for i in xrange(6)]
        self.attrs['mp_male'], self.attrs['mp_female'] = mp_sex_bits
        self.attrs['mp_l18'], self.attrs['mp_18_24'], self.attrs['mp_25_34'], self.attrs['mp_35_44'], \
            self.attrs['mp_45_54'], self.attrs['mp_g55'] = mp_age_bits
        self.attrs['mp_align_bit_no_meaning'] = 0       # bit added specially for pshevtsov@

        self.bitstring = self.make_bitstring()

    def make_bitstring(self):
        res = ''
        for attr in self.attr_meta:
            attr_code = attr[1]
            attr_len = attr[2]
            if attr_len != -1:
                res += ('{:0' + str(attr_len) + 'b}').format(self.attrs[attr_code])
            else:
                for a in self.attrs[attr_code]:
                    a_bin = '{:012b}'.format(int(a)).strip()
                    if len(a_bin) > 12:
                        raise ValueError('Too big attr_value: ' + a)
                    res += a_bin

        # remove trailing zero bytes
        res.rstrip('0')

        # add zeroes for full byte at the end
        while len(res) % 8 != 0:
            res += '0'

        return res

    def id(self):
        return self.id

    def string(self):
        return str(bytearray(self.bytes()))

    def base64(self):
        return base64.standard_b64encode(self.string())

    def base64_bytes(self):
        return [b for b in bytearray(self.base64())]

    def base64_binary(self, bytes_separator=''):
        res = []
        for b in self.base64_bytes():
            res.append('{:08b}'.format(b))
        return bytes_separator.join(res)

    def bytes(self):
        bs = []
        s = '{:032b}'.format(self.id) + self.bitstring
        for i in range(0, len(s), 8):
            bs.append(int(s[i:i + 8], 2))
        return bs

    def tns(self):
        s = self.bitstring
        return s[2:6] + {'00': '0', '01': 'A', '10': 'B', '11': 'C'}[s[6:8]]
