import datetime
import os
import sys
import json
import requests

import toloka
import toloka.api as ta
import yt.wrapper as yt

from rtcconf import config
from utils import utils

sys.path.append(os.environ.get('GRAPH_PY_FOLDER'))
sys.path.append(os.path.join("..", ".."))


class TolokaTask(object):
    def __init__(self, name, prjid, size, max_duration, reward):
        self.name = name
        self.project_id = prjid
        self.size = size
        self.max_duration = max_duration
        self.reward = reward


if config.CRYPTA_ENV == 'testing':
    CROSSDEVICE_TASK = TolokaTask('cross_device', '9038', 3, 900, 0.05)
    INDEVICE_MOBILE_TASK = TolokaTask('indevice_mobile', '9037', 3, 900, 0.03)
    INDEVICE_DESKTOP_TASK = TolokaTask('indevice_desktop', '9039', 3, 540, 0.03)
    HOUSEHOLD_TASK = TolokaTask('household', '9539', 3, 900, 0.1)
else:
    CROSSDEVICE_TASK = TolokaTask('cross_device', '10971', 300, 900, 0.05)
    INDEVICE_MOBILE_TASK = TolokaTask('indevice_mobile', '10970', 250, 900, 0.03)
    INDEVICE_DESKTOP_TASK = TolokaTask('indevice_desktop', '10969', 250, 540, 0.03)
    HOUSEHOLD_TASK = TolokaTask('household', '11765', 280, 3600, 0.07)

TOLOKA_BACKEND_URL = unicode(config.TOLOKA_BACKEND_URL)


def mkclient():
    return ta.Client(config.TOLOKA_ACCESS_TOKEN, 'https://' + config.TOLOKA_HOST)


def create_pool(toloka_task, dt):
    client = mkclient()
    filters = {
        CROSSDEVICE_TASK.name: {u'and': [
            {u'or': [{u'category': u'profile', u'operator': u'EQ', u'value': u'RU', u'key': u'citizenship'}]},
            {u'or': [{u'category': u'computed', u'operator': u'EQ', u'value': u'BROWSER', u'key': u'user_agent_type'}]}]
        },
        INDEVICE_MOBILE_TASK.name: {u'and': [
            {u'or': [{u'category': u'profile', u'operator': u'EQ', u'value': u'RU', u'key': u'country'}]},
            {u'or': [{u'category': u'computed', u'key': u'client_type', u'operator': u'EQ', u'value': u'TOLOKA_APP'}]}]
        },
        INDEVICE_DESKTOP_TASK.name: {u'and': [
            {u'or': [{u'category': u'profile', u'operator': u'EQ', u'value': u'RU', u'key': u'country'}]},
            {u'or': [{u'category': u'computed', u'operator': u'EQ', u'value': u'BROWSER', u'key': u'user_agent_type'}]},
            {u'or': [{u'category': u'computed', u'operator': u'NE', u'value': u'MOBILE_BROWSER', u'key': u'user_agent_type'}]}]
        },
        HOUSEHOLD_TASK.name: {u'and': [
            {u'or': [{u'category': u'profile', u'operator': u'EQ', u'value': u'RU', u'key': u'citizenship'}]},
            {u'or': [{u'category': u'computed', u'operator': u'EQ', u'value': u'BROWSER', u'key': u'user_agent_type'}]}]
        }
    }

    newpool = client.pools.create(
        project_id=toloka_task.project_id,
        private_name='%s_%s_%d' % (toloka_task.name, dt.strftime('%Y-%m-%d'), toloka_task.size),
        assignment_max_duration_seconds=toloka_task.max_duration,
        auto_accept_solutions=False,
        reward_per_assignment=toloka_task.reward,
        may_contain_adult_content=False,
        auto_close_after_complete_delay_seconds=0,
        will_expire=dt + datetime.timedelta(days=200),
        filter=filters[toloka_task.name],
        defaults={"default_overlap_for_new_task_suites": toloka_task.size}
    )

    return newpool


def create_task_suite(toloka_task, pool_id):
    suite_tasks = {
        CROSSDEVICE_TASK.name: [
            {u'input_values': {u'url': TOLOKA_BACKEND_URL + '/start?taskId='}}
        ],
        INDEVICE_MOBILE_TASK.name: [
            {u'input_values': {u'url': TOLOKA_BACKEND_URL + '/m/firefox/code?taskId=', u'browser': u'Firefox'}},
            {u'input_values': {u'url': TOLOKA_BACKEND_URL + '/m/chrome/code?taskId=', u'browser': u'Chrome'}},
            {u'input_values': {u'url': TOLOKA_BACKEND_URL + '/m/safari/code?taskId=', u'browser': u'Safari'}},
            {u'input_values': {u'url': TOLOKA_BACKEND_URL + '/m/yandex/code?taskId=', u'browser': u'Yandex Browser'}},
            {u'input_values': {u'url': TOLOKA_BACKEND_URL + '/m/android/code?taskId=', u'browser': u'Android Browser'}},
            {u'input_values': {u'url': TOLOKA_BACKEND_URL + '/m/uc/code?taskId=', u'browser': u'UC Browser'}},
            {u'input_values': {u'url': TOLOKA_BACKEND_URL + '/m/opera/code?taskId=', u'browser': u'Opera'}},
            {u'input_values': {u'url': TOLOKA_BACKEND_URL + '/m/dolfin/code?taskId=', u'browser': u'Dolphin'}},
            {u'input_values': {u'url': TOLOKA_BACKEND_URL + '/m/samsung/code?taskId=', u'browser': u'Samsung Internet'}}
        ],
        INDEVICE_DESKTOP_TASK.name: [
            {u'input_values': {u'url': TOLOKA_BACKEND_URL + u'/firefox/code?taskId=', u'browser': u'Firefox'}},
            {u'input_values': {u'url': TOLOKA_BACKEND_URL + u'/chrome/code?taskId=', u'browser': u'Chrome'}},
            {u'input_values': {u'url': TOLOKA_BACKEND_URL + u'/safari/code?taskId=', u'browser': u'Safari'}},
            {u'input_values': {u'url': TOLOKA_BACKEND_URL + u'/yandex/code?taskId=', u'browser': u'Yandex Browser'}},
            {u'input_values': {u'url': TOLOKA_BACKEND_URL + u'/ie/code?taskId=', u'browser': u'Internet Explorer/Microsoft Edge'}}
        ],
        HOUSEHOLD_TASK.name: [
            {u'input_values': {u'url': TOLOKA_BACKEND_URL + '/start?taskId='}}
        ],
    }

    client = mkclient()
    return client.task_suites.create(
        infinite_overlap=False,
        tasks=suite_tasks[toloka_task.name],
        pool_id=unicode(pool_id),
        overlap=toloka_task.size,
        mixed=False
    )


if '__main__' == __name__:
    try:
        yt.config.set_proxy(config.MR_SERVER)

        all_task_types = [CROSSDEVICE_TASK, INDEVICE_DESKTOP_TASK, INDEVICE_MOBILE_TASK, HOUSEHOLD_TASK]
        selected_task_types = []

        if len(sys.argv) >= 2:
            for tt in all_task_types:
                if tt.name == sys.argv[1]:
                    selected_task_types.append(tt)

            if not selected_task_types:
                print 'Unknown task type: \'%s\'' % sys.argv[1]
                sys.exit(1)
        else:
            selected_task_types = all_task_types

        new_pools = []
        for task in selected_task_types:
            pool = create_pool(task, datetime.datetime.now())
            new_pools.append((task, pool))

        for (task, pool) in new_pools:
            create_task_suite(task, pool.id)

        client = mkclient()
        for (task, pool) in new_pools:
            item = toloka.pools.Item(client.pools, pool.id, toloka.pools.Entity)
            item.open()

        pools_table = os.path.join(config.CRYPTA_TOLOKA_FOLDER, 'pools')
        yt.write_table(yt.TablePath(pools_table, append=True), [
            dict(pool_id=p[1].id, pool_status='running', task_name=p[0].name)
            for p in new_pools
        ])

        utils.monrun_ok_nodate()
    except:
        utils.monrun_error_nodate()
        raise
