#!/usr/bin/python
# coding=utf-8
import logging
import os
import time
from datetime import datetime

import luigi
import yt.wrapper as yt

from lib.luigi import base_luigi_task
from rtcconf import config
from utils import mr_utils as mr


def reset_global_yt_state():
    logging.debug('Patching yt client state')
    yt._cleanup_http_session()


class YtTarget(luigi.Target):

    def __init__(self, table, allow_empty=False):
        self.table = table
        self.allow_empty = allow_empty

    def exists(self):
        reset_global_yt_state()
        if yt.exists(self.table):
            return self.allow_empty or (mr.row_count(self.table) > 0) or getattr(config, 'YT_TARGET_EMPTY_TABLE_OK', False)
        else:
            return False


class YtFolderTarget(luigi.Target):
    def __init__(self, folder, allow_empty=False, absolute_path=False):
        self.folder = folder
        self.allow_empty = allow_empty
        self.absolute_path = absolute_path

    def exists(self):
        reset_global_yt_state()
        if not yt.exists(self.folder):
            return False
        if not self.allow_empty:
            return len(mr.ls(self.folder, absolute_path=self.absolute_path)) > 0
        else:
            return True


class YtDateTarget(luigi.Target):
    def __init__(self, table, date):
        self.table = table
        self.date = date

    def exists(self):
        reset_global_yt_state()
        if yt.exists(self.table):
            generate_date = mr.get_generate_date(self.table)
            return self.date <= generate_date
        else:
            return False


class YtDateColumnTarget(YtDateTarget):
    def __init__(self, table, column, date):
        super(YtDateColumnTarget, self).__init__(table, date)
        self.column = column

    def exists(self):
        reset_global_yt_state()
        table_exists = super(YtDateColumnTarget, self).exists()
        return table_exists and self.column in yt.read_table(self.table,
                                                             format='json',
                                                             raw=False).next()


class FileTarget(luigi.Target):
    def __init__(self, path):
        self.path = path

    def exists(self):
        reset_global_yt_state()
        return yt.exists(self.path)


class FilePrepared(luigi.ExternalTask):
    yt_path = luigi.Parameter()

    def output(self):
        return FileTarget(self.yt_path)


class TodayFileTarget(luigi.Target):
    def __init__(self, filename, date):
        self.filename = filename + '.date'
        self.date = date

    @staticmethod
    def done(filename, date):
        filename += '.date'
        with open(filename, 'w') as fw:
            fw.write(date)

    def exists(self):
        if not os.path.exists(self.filename):
            return False
        with open(self.filename, 'r') as fr:
            existing_date = fr.readline().strip()
            return self.date <= existing_date


class YtAttrTarget(luigi.Target):

    def __init__(self, path, attr):
        self.path = path
        self.attr = attr

    def exists(self):
        reset_global_yt_state()
        return yt.exists(self.path) and bool(mr.safe_get_attribute(self.path, self.attr))

    def complete(self):
        yt.set_attribute(self.path, self.attr, True)


class ExternalInput(luigi.ExternalTask):

    table = luigi.Parameter()
    allow_empty = luigi.Parameter(default=False)

    def output(self):
        return YtTarget(self.table, self.allow_empty)


class ExternalInputLastDate(luigi.ExternalTask):

    folder = luigi.Parameter()

    def output(self):
        fail_count = 0
        last_error = None
        while fail_count < 10:
            try:
                table = mr.get_last_table(self.folder)
                return YtTarget(table)
            except Exception as e:
                fail_count += 1
                last_error = e
                time.sleep(5)

        raise last_error


class YesterdayDictInput(ExternalInput):
    def __init__(self, table_name):
        table = config.GRAPH_YT_DICTS_FOLDER + table_name
        super(YesterdayDictInput, self).__init__(table)


class PostGraphTask(base_luigi_task.BaseTask):
    date = luigi.Parameter()
    name = luigi.Parameter()

    def run(self):
        self.run_post_graph()
        set_current_time_attr(self.path(), self.name)

    def run_post_graph(self):
        pass

    def output(self):
        return YtAttrTarget(self.path(), self.name)

    def path(self):
        return config.YT_OUTPUT_FOLDER + self.date


class BaseYtTask(base_luigi_task.BaseTask):

    def __init__(self, *args, **kwargs):
        super(BaseYtTask, self).__init__(*args, **kwargs)

    def input_folders(self):
        raise NotImplementedError

    def workdir(self):
        raise Exception('No workdir set')

    def output_folders(self):
        raise NotImplementedError

    def in_f(self, folder_alias):
        """
        Get input folder with specific alias from input_folders mapping
        """
        return self.input_folders()[folder_alias]

    def out_f(self, folder_alias):
        """
        Get output folder with specific alias from output_folders mapping
        """
        return self.output_folders()[folder_alias]


def run_task(task_name, date, memory_limit=1024):
    yt.config.set_proxy(config.MR_SERVER)
    luigi.run([task_name, '--date', date, '--workers', '10', '--local-scheduler'])


def set_current_time_attr(path, attr):
    dt = datetime.now().strftime('%Y-%m-%d %H:%M:%S')
    yt.set_attribute(path, attr, dt)
