import phonenumbers
import re


PHONE_PREFIXES = {'BEELINE': ['903', '905', '906', '909', '951', '953', '960', '961', '962', '964', '965', '966',
                              '967', '968'],
                  'MTS': ['910', '911', '912', '913', '914', '915', '916', '917', '918', '919', '980', '981', '982',
                          '983', '984', '985', '987', '988', '989'],
                  'MEGAFON': ['920', '921', '922', '924', '925', '926', '927', '928', '929'],
                  'TELE2': ['900', '901', '902', '904', '908', '950', '951', '952', '953', '958', '977', '991', '992',
                            '993', '994', '995', '996']}

PHONE_PREFIXES_REVERSE_DICT = {
    prefix: op
    for op, prefixes in PHONE_PREFIXES.iteritems()
    for prefix in prefixes
}


def init():
    """
    This method should be called prior to using phone_parser in YT jobs
    It inits all metadata, which will not be available for dynamic load at YT machines
    """
    phonenumbers.PhoneMetadata.load_all()


def parse_phone_number(number_str):
    """Parse phone number
    
    :param number_str: phone number
    :return: (formatted_number, region_code)
    :rtype: tuple[str|None, str|None]
    """
    def parse_international_format(phone_symbols_only, county_code=None):
        parsed_number = phonenumbers.parse(phone_symbols_only, county_code)
        # if phonenumbers.is_possible_number(parsed_number):
        if phonenumbers.is_valid_number(parsed_number):
            formatted_number = phonenumbers.format_number(parsed_number, phonenumbers.PhoneNumberFormat.E164)
            region_code = phonenumbers.region_code_for_country_code(parsed_number.country_code)
            return formatted_number, region_code
        else:
            return None

    phone_symbols_only = re.sub('[^\\d+]', '', number_str)

    if not phone_symbols_only:
        return None

    try:
        if phone_symbols_only.startswith('+'):
            return parse_international_format(phone_symbols_only)
        elif phone_symbols_only.startswith('8') or phone_symbols_only.startswith('7'):
            # common Russian and Kazakhstanian case and some stupid Ukrainian case
            phone = parse_international_format(phone_symbols_only, 'RU')
            if not phone:
                phone = parse_international_format(phone_symbols_only, 'KZ')
            if not phone and phone_symbols_only.startswith('8'):
                phone = parse_international_format(phone_symbols_only, 'BY')
            if not phone and phone_symbols_only.startswith('8'):  # why do they write phone like that?
                phone = parse_international_format('3' + phone_symbols_only, 'UA')
            return phone
        elif phone_symbols_only.startswith('38') or (phone_symbols_only.startswith('0')
                                                     and len(phone_symbols_only) == 10):
            # common Ukranian case
            return parse_international_format(phone_symbols_only, 'UA')
        elif phone_symbols_only.startswith('375'):
            # common Belarus case
            return parse_international_format(phone_symbols_only, 'BY')
        elif phone_symbols_only.startswith('9') and len(phone_symbols_only) == 10:
            # russian with no country code
            return parse_international_format('+7' + phone_symbols_only, 'RU')
        elif phone_symbols_only.startswith('90'):
            # common Turkey case
            return parse_international_format(phone_symbols_only, 'TR')

        else:
            # just try to make iternational number
            return parse_international_format('+' + phone_symbols_only)
    except phonenumbers.phonenumberutil.NumberParseException:
        return None


def parse_phone_numbers(number_str):
    if len(number_str) > 20:
        several_phones = []
        number_str = number_str.decode('unicode_escape').encode('ascii', errors='ignore')  # cut all non ascii
        for match in phonenumbers.PhoneNumberMatcher(number_str, 'RU'):  # support only the most common case
            phone_n = phonenumbers.format_number(match.number, phonenumbers.PhoneNumberFormat.E164)
            several_phones.append((phone_n, 'RU'))
        return several_phones
    else:
        single_number_and_region = parse_phone_number(number_str)
        if single_number_and_region:
            return [single_number_and_region]
        else:
            return []
