import inspect
import logging
import sys
import uuid
from logging.handlers import SysLogHandler

import yt.wrapper as yt

from utils import logging_setup

logger = logging.getLogger('yt_trace')


class YtOperationWrapper(object):
    def __init__(self, op, execution_id):
        self.__dict__['op'] = op
        self.__dict__['execution_id'] = execution_id

    def get_state(self):
        result = self.op.get_state()
        logger.debug('%s||%s||%s', self.execution_id, self.op.id, result)
        return result

    def __getattr__(self, item):
        return getattr(self.op, item)

    def __setattr__(self, item, val):
        return setattr(self.op, item, val)


def wrap_yt_run(func):
    def wrapper(*args, **kwargs):
        fi = inspect.getframeinfo(inspect.currentframe().f_back)
        sync = kwargs.get('sync', True)
        execution_id = uuid.uuid4().hex
        logger.debug('%s||start||%s||%s||%s||%s||(%s@%s:%s)', execution_id, sync, func.__name__, repr(args), repr(kwargs), fi.function, fi.filename, fi.lineno)
        try:
            res = func(*args, **kwargs)
            op_id = 'unknown'
            if res is not None and not sync:
                res = YtOperationWrapper(res, execution_id)
                op_id = res.id
            logger.debug('%s||end||%s||%s||%s||%s||(%s@%s:%s)||%s', execution_id, sync, func.__name__, repr(args), repr(kwargs), fi.function, fi.filename, fi.lineno, op_id)
            return res
        except Exception as e:
            logger.error('%s||error||%s||%s||%s||%s||(%s@%s:%s)||%s', execution_id, sync, func.__name__, repr(args), repr(kwargs), fi.function, fi.filename, fi.lineno, str(e))
            exc_info = sys.exc_info()
            raise exc_info[0], exc_info[1], exc_info[2]

    return wrapper


def setup_trace():
    yt.run_map = wrap_yt_run(yt.run_map)
    yt.run_reduce = wrap_yt_run(yt.run_reduce)
    yt.run_map_reduce = wrap_yt_run(yt.run_map_reduce)
    yt.run_sort = wrap_yt_run(yt.run_sort)
    yt.run_merge = wrap_yt_run(yt.run_merge)

    logger.setLevel(logging.DEBUG)
    handler = SysLogHandler(address='/dev/log', facility='local4')
    handler.setLevel(logging.DEBUG)
    handler.setFormatter(logging.Formatter(logging_setup.THE_LOG_FORMAT))
    logger.addHandler(handler)
