import json
from functools import partial

import luigi
import yt.wrapper as yt

from account_manager_decoder.decoder import decode
from lib.luigi import yt_luigi
from rtcconf import config
from utils import mr_utils as mr
from v2 import ids
from v2.soup import soup_config
from v2.soup.soup_tables import SoupDumpTable

AM_ACCOUNT_TYPES = {
    config.ID_TYPE_PHONE: 'a',
    config.ID_TYPE_EMAIL: 'b'
}

OOM_LIMIT = 100


def parse_am_events(rec, date):
    def decode_account_events(account_type):
        event_name = rec.get('EventName', '')
        if event_name == 'AM_System AM info':
            try:
                encrypted = json.loads(rec.get('EventValue', ''))['a']
                mmetric_device_id = rec.get('DeviceID', '')
                decrypted = decode(encrypted, mmetric_device_id)
                decrypted_unpadded = decrypted[:decrypted.rfind('}') + 1]

                emails = set()
                acc_pairs = json.loads(decrypted_unpadded)['a']
                for email in [x[0][1:] for x in acc_pairs if x[0].startswith(AM_ACCOUNT_TYPES[account_type])]:
                    emails.add(email)

                for email in emails:
                    yield email
            except:
                pass

    device_id = rec[ids.CRYPTA_DEVICE_ID]

    ts = int(rec.get('_logfeller_timestamp', rec.get('timestamp')))

    for email in decode_account_events(config.ID_TYPE_EMAIL):
        yield {'id_hash': email, 'id_type': config.ID_TYPE_EMAIL, 'source_type': config.ID_SOURCE_TYPE_ACCOUNT_MANAGER,
               'deviceid': device_id, 'ts': ts, 'date': date, '@table_index': 0}

    for phone in decode_account_events(config.ID_TYPE_PHONE):
        yield {'id_hash': phone, 'id_type': config.ID_TYPE_PHONE, 'source_type': config.ID_SOURCE_TYPE_ACCOUNT_MANAGER,
               'deviceid': device_id, 'ts': ts, 'date': date, '@table_index': 1}


def merge_day_to_dict(key, recs):
    history = set()
    for r in recs:
        existing_history = r.get('history')
        if existing_history:
            history.update(set(existing_history))

        new_date = r.get('date')
        if new_date:
            history.add(new_date)

    if history:
        history = sorted(history)
        last_date = history[-1]
    else:
        history = []
        last_date = None

    out_rec = dict(key)
    out_rec['history'] = history
    out_rec['last_date'] = last_date
    yield out_rec


def map_am_to_soup(rec):
    dates = rec.get('history')
    if dates:
        if rec['id_type'] == 'email':
            # device_id -- in this context mean always google_adv_id
            yield SoupDumpTable.make_rec(rec['deviceid'], rec['id_hash'], soup_config.gaid_email_am, dates, 0)
        elif rec['id_type'] == 'phone':
            yield SoupDumpTable.make_rec(rec['deviceid'], rec['id_hash'], soup_config.gaid_phone_am, dates, 1)


class AccountManagerUpdateDictTask(yt_luigi.BaseYtTask):
    date = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(AccountManagerUpdateDictTask, self).__init__(*args, **kwargs)
        self.gaid_email_am = SoupDumpTable(soup_config.gaid_email_am, self.date)
        self.gaid_phone_am = SoupDumpTable(soup_config.gaid_phone_am, self.date)


    def input_folders(self):
        return {
            'dict': config.GRAPH_YT_DICTS_FOLDER,
            'am_day': config.YT_OUTPUT_FOLDER + self.date + '/mobile/account_manager/'
        }

    def output_folders(self):
        return {
            'dict': config.GRAPH_YT_DICTS_FOLDER,
            'am_day': config.YT_OUTPUT_FOLDER + self.date + '/mobile/account_manager/'
        }

    def requires(self):
        from data_imports.import_logs import app_metrica_day
        return app_metrica_day.ImportAppMetrikaDayTask(date=self.date, run_date=self.date)

    def run(self):
        in_am_log_table = self.in_f('am_day') + 'am_log'
        in_am_dict_table = self.in_f('dict') + 'account_manager'

        mr.mkdir(self.out_f('am_day'))
        email_am_day_table = self.out_f('am_day') + 'am_device_md5' + config.ID_TYPE_EMAIL
        phone_am_day_table = self.out_f('am_day') + 'am_device_md5' + config.ID_TYPE_PHONE

        # decoding is CPU-intensive, let's put 1K records per job
        job_count = yt.row_count(in_am_log_table) / 1000 + 1
        yt.run_map(partial(parse_am_events, date=self.date),
                   in_am_log_table,
                   [email_am_day_table, phone_am_day_table],
                   job_count=job_count)

        am_join_tables = [email_am_day_table, phone_am_day_table]
        if mr.exists(in_am_dict_table):
            # if no dict exist, we will just create it, otherwise append to it
            am_join_tables.append(in_am_dict_table)

        am_key = ['id_hash', 'id_type', 'deviceid', 'source_type']
        mr.sort_all(am_join_tables, sort_by=am_key)
        yt.run_reduce(merge_day_to_dict,
                      am_join_tables,
                      self.out_f('am_day') + 'account_manager_dict',  # use day folder for backup
                      reduce_by=am_key)

        mr.mkdir(self.out_f('dict'))
        mr.copy(self.out_f('am_day') + 'account_manager_dict', self.out_f('dict') + 'account_manager')
        mr.set_generate_date(self.out_f('dict') + 'account_manager', self.date)

        with yt.Transaction() as tr:
            yt.run_map(map_am_to_soup,
                       self.in_f('dict') + 'account_manager',
                       [self.gaid_email_am.create(tr),
                        self.gaid_phone_am.create(tr)])

            SoupDumpTable.finalize_all([
                self.gaid_email_am,
                self.gaid_phone_am
            ], tr)

    def output(self):
        soup_targets = [t.as_target() for t in [
            self.gaid_email_am,
            self.gaid_phone_am
        ]]

        return [yt_luigi.YtTarget(self.out_f('am_day') + 'account_manager_dict', self.date),
                yt_luigi.YtDateTarget(self.out_f('dict') + 'account_manager', self.date)] + soup_targets


if __name__ == '__main__':
    yt.config.set_proxy(config.MR_SERVER)

    import smart_runner

    smart_runner.run_isolated('//home/crypta/team/artembelov/am_date/', '2016-06-22',
                              AccountManagerUpdateDictTask,
                              AccountManagerUpdateDictTask, '2016-06-22')
