#!/usr/bin/python
# coding=utf-8

from matching.device_matching.fuzzy_ml.features import TDevFeatures
from utils import mr_utils as mr
from utils import utils, geo_utils
from v2 import ids
from v2.ids_storage import column_aggregators
from v2.soup import soup_config

OOM_LIMIT = 10000

CONNECTION_TYPE_MAPPER = {
    'CONN_WIFI': 'wifi',
    'CONN_CELL': 'cell',
}

# CRYPTADA-613 copied as is from previous impl
class ConnectionHistAggregator(column_aggregators.ColumnAggregator):
    def __init__(self):
        self.connection_hist_timestamps = {conn_type: set() for conn_type in CONNECTION_TYPE_MAPPER.values()}
        self.result_connection_hist = {conn_type: 0 for conn_type in CONNECTION_TYPE_MAPPER.values()}

        super(ConnectionHistAggregator, self).__init__()

    def accept_rec(self, rec):
        # this will be preseneted on month level reduce
        connection_hist_str = rec.get('connection_hist', '')
        # this will be presented on day level reduce
        connection_type = rec.get('connection_type', '')

        if connection_hist_str:
            connection_hist = utils.parse_dict_field(connection_hist_str)
            for conn_type, conn_count in connection_hist.iteritems():
                if conn_type in self.result_connection_hist:
                    self.result_connection_hist[conn_type] += int(conn_count)
        elif connection_type and connection_type in self.result_connection_hist:
            self.connection_hist_timestamps[connection_type].add(rec['timestamp_10m'])

    def get_aggregated_value(self):
        if sum(len(event_timestamps) for event_timestamps in self.connection_hist_timestamps.values()):
            self.result_connection_hist = {conn_type: len(conn_timestamps)
                                           for conn_type, conn_timestamps
                                           in self.connection_hist_timestamps.iteritems()}
        return self.result_connection_hist


class MergeFeaturesAggregator(column_aggregators.ColumnAggregator):
    def __init__(self, column):
        self.column = column
        self.features = TDevFeatures()

    def accept_rec(self, rec):
        feature_value = rec.get(self.column)
        if feature_value:
            self.features.merge(TDevFeatures(feature_value))

    def get_aggregated_value(self):
        return self.features.to_string()


dev_info_info_schema = {
    'id': 'string',
    'id_type': 'string',
    'source': 'string',

    ids.CRYPTA_DEVICE_ID: 'string',
    'mmetric_device_ids_to_hash': 'any',
    ids.IDFA: 'string',
    ids.ANDROID_ID: 'string',
    ids.GOOGLE_ADV_ID: 'string',
    'mmetric_device_ids': 'any',

    'apps': 'any',
    'metrika_apps': 'any',
    'ssp_apps': 'any',

    'device_type': 'string',
    'manufacturer': 'string',
    'model': 'string',
    'os': 'string',
    'os_version': 'string',
    'ua_profile': 'string',
    'locale': 'string',
    'connection_hist': 'any',

    'features': 'any',
    'macs': 'any',

    'region_ids': 'any',
    'main_region_country': 'uint64',
    'main_region_obl': 'uint64',
    'main_region_city': 'uint64',
    'main_region': 'uint64',

    'ts': 'int64',
    'dates': 'any',
    'subkey': 'string'
}

def reduce_dev_info_yt(key, recs, is_day_task):
    try:
        first_rec_aggr = column_aggregators.FirstRecAggregator()

        metrika_apps_agg = column_aggregators.SetUnionAggregator('metrika_apps', oom_limit=OOM_LIMIT)
        mac_aggr = column_aggregators.SetUnionAggregator("macs")

        connection_hist_agg = ConnectionHistAggregator()
        features_agg = MergeFeaturesAggregator(column='features')

        if is_day_task:
            dates_agg = column_aggregators.SetUnionAggregator('date', oom_limit=OOM_LIMIT)
            ssp_apps_agg = column_aggregators.SetUnionAggregator('ssp_app', oom_limit=OOM_LIMIT)
            region_ids_agg = column_aggregators.MergeHitsDictsAggregator('region_id')
            mmetric_device_ids_agg = column_aggregators.SetUnionAggregator(ids.MMETRIC_DEVICE_ID, oom_limit=OOM_LIMIT)
            mmetric_device_ids_to_hash_agg = column_aggregators.NewKeyValueMappingAggregator(
                ids.MMETRIC_DEVICE_ID, ids.MMETRIC_DEVICE_ID + ids.HASH_POSTFIX, oom_limit=OOM_LIMIT
            )

        else:
            dates_agg = column_aggregators.SetUnionAggregator('dates', oom_limit=OOM_LIMIT)
            ssp_apps_agg = column_aggregators.SetUnionAggregator('ssp_apps', oom_limit=OOM_LIMIT)
            region_ids_agg = column_aggregators.MergeHitsDictsAggregator('region_ids')
            mmetric_device_ids_agg = column_aggregators.SetUnionAggregator('mmetric_device_ids', oom_limit=OOM_LIMIT)
            mmetric_device_ids_to_hash_agg = column_aggregators.MergeDictAggregator(
                'mmetric_device_ids_to_hash', oom_limit=OOM_LIMIT
            )


        aggregators = [
            first_rec_aggr,
            mmetric_device_ids_agg,
            metrika_apps_agg,
            ssp_apps_agg,
            mac_aggr,
            connection_hist_agg,
            features_agg,
            region_ids_agg,
            dates_agg,
            mmetric_device_ids_to_hash_agg
        ]

        for rec in recs:
            for aggregator in aggregators:
                aggregator.accept_rec(rec)

        out_rec = first_rec_aggr.get_aggregated_value()

        out_rec['id'] = out_rec[ids.CRYPTA_DEVICE_ID]
        out_rec['id_type'] = ids.CRYPTA_DEVICE_ID
        out_rec['source'] = soup_config.LOG_SOURCE_MOBILE_METRIKA

        out_rec['metrika_apps'] = sorted(metrika_apps_agg.get_aggregated_value())
        out_rec['ssp_apps'] = sorted(ssp_apps_agg.get_aggregated_value())

        all_apps = set(out_rec['metrika_apps']).union(set(out_rec['ssp_apps']))
        out_rec['apps'] = sorted(all_apps)

        out_rec['mmetric_device_ids'] = mmetric_device_ids_agg.get_aggregated_value()
        out_rec['mmetric_device_ids_to_hash'] = mmetric_device_ids_to_hash_agg.get_aggregated_value()
        out_rec['dates'] = dates_agg.get_aggregated_value()
        out_rec['connection_hist'] = utils.convert_to_dict_field(connection_hist_agg.get_aggregated_value())
        out_rec['macs'] = mac_aggr.get_aggregated_value()
        out_rec['features'] = features_agg.get_aggregated_value()

        region_ids = region_ids_agg.get_aggregated_value()
        out_rec['region_ids'] = region_ids_agg.get_aggregated_value()
        country_id, oblast_id, city_id, main_region = geo_utils.get_top_by_region_level(region_ids)
        out_rec['main_region_country'] = country_id
        out_rec['main_region_obl'] = oblast_id
        out_rec['main_region_city'] = city_id
        out_rec['main_region'] = main_region

        out_rec['@table_index'] = 0
        yield out_rec

    except mr.OomLimitException:
        device_id = key.get(ids.CRYPTA_DEVICE_ID) or key.get("id")
        yield {ids.CRYPTA_DEVICE_ID: device_id, '@table_index': 1}
        return



uuid_info_schema = {
    'id': 'string',
    'id_type': 'string',
    'source': 'string',

    ids.UUID: 'string',
    'api_keys': 'any',
    'app_id': 'string',
    'app_version': 'string',

    ids.CRYPTA_DEVICE_ID: 'string',
    ids.MMETRIC_DEVICE_ID + ids.HASH_POSTFIX: 'string',
    ids.IDFA: 'string',
    ids.ANDROID_ID: 'string',
    ids.GOOGLE_ADV_ID: 'string',
    ids.MMETRIC_DEVICE_ID: 'string',
    'ua_profile': 'string',

    'dates': 'any',
    'ts': 'int64',
    'subkey': 'string',
}
def reduce_uuid_info(uuid_key, recs, is_day_task):
    try:
        uuid_agg = column_aggregators.FirstValueAggregator(ids.UUID)
        app_id_agg = column_aggregators.FirstValueAggregator('app_id')
        app_version_agg = column_aggregators.FirstValueAggregator('app_version')

        devid_agg = column_aggregators.FirstValueAggregator(ids.CRYPTA_DEVICE_ID)
        mmetric_devid_agg = column_aggregators.FirstValueAggregator(ids.MMETRIC_DEVICE_ID)
        mmetric_devid_hash_agg = column_aggregators.FirstValueAggregator(ids.MMETRIC_DEVICE_ID + ids.HASH_POSTFIX)
        idfa_agg = column_aggregators.FirstValueAggregator(ids.IDFA)
        google_adv_id_agg = column_aggregators.FirstValueAggregator(ids.GOOGLE_ADV_ID)
        android_id_agg = column_aggregators.FirstValueAggregator(ids.ANDROID_ID)
        ua_profile_agg = column_aggregators.FirstValueAggregator('ua_profile')

        ts_agg = column_aggregators.FirstValueAggregator('ts')

        if is_day_task:
            api_keys_agg = column_aggregators.MergeHitsDictsAggregator('api_key')
            dates_agg = column_aggregators.SetUnionAggregator('date', oom_limit=OOM_LIMIT)
        else:
            api_keys_agg = column_aggregators.MergeHitsDictsAggregator('api_keys')
            dates_agg = column_aggregators.SetUnionAggregator('dates', oom_limit=OOM_LIMIT)

        aggregators = [
            uuid_agg,
            app_id_agg,
            app_version_agg,
            devid_agg,
            mmetric_devid_agg,
            mmetric_devid_hash_agg,
            idfa_agg,
            google_adv_id_agg,
            android_id_agg,
            ua_profile_agg,
            ts_agg,
            api_keys_agg,
            dates_agg
        ]

        for rec in recs:
            for aggregator in aggregators:
                aggregator.accept_rec(rec)

        yield {
            'id': uuid_agg.get_aggregated_value(),
            'id_type': ids.UUID,
            'source': soup_config.LOG_SOURCE_MOBILE_METRIKA,

            ids.UUID: uuid_agg.get_aggregated_value(),
            'app_id': app_id_agg.get_aggregated_value(),
            'app_version': app_version_agg.get_aggregated_value(),

            ids.CRYPTA_DEVICE_ID: devid_agg.get_aggregated_value(),
            ids.MMETRIC_DEVICE_ID: mmetric_devid_agg.get_aggregated_value(),
            ids.MMETRIC_DEVICE_ID + ids.HASH_POSTFIX: mmetric_devid_hash_agg.get_aggregated_value(),
            ids.IDFA: idfa_agg.get_aggregated_value(),
            ids.GOOGLE_ADV_ID: google_adv_id_agg.get_aggregated_value(),
            ids.ANDROID_ID: android_id_agg.get_aggregated_value(),
            'ua_profile': ua_profile_agg.get_aggregated_value(),

            'ts': ts_agg.get_aggregated_value(),
            'api_keys': api_keys_agg.get_aggregated_value(),
            'dates': dates_agg.get_aggregated_value()
        }

    except mr.OomLimitException:
        yield {'uuid': uuid_key['uuid'], '@table_index': 1}
        return
