from functools import partial

import luigi
import yt.wrapper as yt

import app_metrica_dict_schemas
import app_metrica_month
from data_imports.import_logs import app_metrica_day
from lib.luigi import yt_luigi
from rtcconf import config
from utils import mr_utils as mr
from utils import utils
from v2 import ids


def to_mac_address(rec):
    for mac in rec['macs']:
        yield {'devid': rec[ids.CRYPTA_DEVICE_ID], 'mac': mac, 'ts': rec['ts']}


def to_mmetric_hash(rec):
    for mmetric_devid, mmetric_devid_hash in rec['mmetric_device_ids_to_hash'].iteritems():
        yield {
            'devid': rec[ids.CRYPTA_DEVICE_ID],
            'mmetric_devid': mmetric_devid,
            'devidhash': mmetric_devid_hash
        }


def to_yamr_format(rec_key, rec):
    yamr_value = '\t'.join([k + '=' + (str(v) if v else '') for k, v in rec.iteritems()])
    return {'key': rec_key, 'subkey': '', 'value': yamr_value}


def to_dev_info_format(rec, only_yamr_rec=False):
    out_rec = {
        'devid': rec[ids.CRYPTA_DEVICE_ID],
        'idfa': rec[ids.IDFA],
        'android_id': rec[ids.ANDROID_ID],
        'google_adv_id': rec[ids.GOOGLE_ADV_ID],
        'mmetric_devids': ','.join(rec['mmetric_device_ids']),

        'apps': ','.join(rec['apps']),
        'ssp_apps': ','.join(rec['ssp_apps']),

        'device_type': rec['device_type'],
        'manufacturer': rec['manufacturer'],
        'model': rec['model'],
        'os': rec['os'],
        'os_version': rec['os_version'],
        'ua_profile': rec['ua_profile'],
        'locale': rec['locale'],
        'connection_hist': rec['connection_hist'],

        'ts': rec['ts'],
        'dates': rec['dates'],
        'subkey': rec['subkey']
    }

    yamr_rec = to_yamr_format(rec[ids.CRYPTA_DEVICE_ID], out_rec)

    if only_yamr_rec:
        yield yamr_rec

    else:
        yield out_rec

        yamr_rec['@table_index'] = 1
        yield yamr_rec


def to_uuid_info_format(rec, only_yamr_rec=False):
    out_rec = {
        'uuid': rec[ids.UUID],
        'api_keys': rec['api_keys'],
        'app_id': rec['app_id'],
        'app_version': rec['app_version'],

        'devid': rec[ids.CRYPTA_DEVICE_ID],
        'devidhash': rec[ids.MMETRIC_DEVICE_ID + ids.HASH_POSTFIX],
        'idfa': rec[ids.IDFA],
        'android_id': rec[ids.ANDROID_ID],
        'google_adv_id': rec[ids.GOOGLE_ADV_ID],
        'mmetric_devid': rec[ids.MMETRIC_DEVICE_ID],

        'ua_profile': rec['ua_profile'],

        'ts': rec['ts'],
        'subkey': rec['subkey'],
    }

    yamr_rec = to_yamr_format(rec[ids.UUID], out_rec)

    if only_yamr_rec:
        yield yamr_rec

    else:
        yield out_rec

        yamr_rec['@table_index'] = 1
        yield yamr_rec


def reduce_first_rec(_, recs):
    for rec in recs:
        rec['@table_index'] = 0
        yield rec
        return


class ConvertAppMetricaDictsToOldFormats(yt_luigi.BaseYtTask):
    date = luigi.Parameter()

    def input_folders(self):
        return {
            'ids_storage': config.CRYPTA_IDS_STORAGE,
        }

    def output_folders(self):
        return {
            'dict': config.GRAPH_YT_DICTS_FOLDER,
            'mobile': config.GRAPH_YT_OUTPUT_FOLDER + self.date + '/mobile/',
        }

    def requires(self):
        return app_metrica_month.AppMetricaDictMergeMonthTask(self.date)

    def run(self):
        with yt.Transaction() as tr:
            for t in ['devid_hash', 'devid_mac', 'dev_info_yt', 'uuid_dev_info_yt']:
                mr.create_table_with_schema(
                    self.out_f('dict') + t,
                    app_metrica_dict_schemas.dict_table_schemas[t],
                    tr,
                    strict=True,
                    recreate_if_exists=True,
                    # sorted_by=app_metrica_dict_schemas.dict_schemas_sort_keys[t]
                )

            utils.wait_all([
                yt.run_map_reduce(
                    to_mmetric_hash,
                    partial(mr.distinct_mr, distinct_keys=['mmetric_devid', 'devidhash', 'devid']),
                    self.in_f('ids_storage') + 'device_id/app_metrica_month',
                    self.out_f('dict') + 'devid_hash',
                    reduce_by=['devidhash', 'mmetric_devid', 'devid'], sync=False),

                yt.run_map_reduce(
                    to_mac_address,
                    reduce_first_rec,
                    self.in_f('ids_storage') + 'device_id/app_metrica_month',
                    self.out_f('mobile') + 'devid_mac_month_tmp',
                    sort_by=['mac', 'devid', 'ts'],
                    reduce_by=['mac', 'devid'],
                    sync=False),

                yt.run_map(to_dev_info_format,
                           self.in_f('ids_storage') + 'device_id/app_metrica_month',
                           [self.out_f('dict') + 'dev_info_yt',
                            self.out_f('dict') + 'dev_info'],
                           ordered=True,
                           sync=False),

                yt.run_map(to_uuid_info_format,
                           self.in_f('ids_storage') + 'uuid/app_metrica_month',
                           [self.out_f('dict') + 'uuid_dev_info_yt',
                            self.out_f('dict') + 'uuid_dev_info'],
                           ordered=True,
                           sync=False),
            ])

            # devid-mac should be 1-1 relationship to use it in matching
            from matching.device_matching.perfect.enrich import devid_mac_vmetro
            yt.run_map_reduce(None, devid_mac_vmetro.filter_multi_devid_macs,
                              self.out_f('mobile') + 'devid_mac_month_tmp',
                              [self.out_f('dict') + 'devid_mac',
                               self.out_f('mobile') + 'mac_multi_devid'],
                              reduce_by='mac')

            utils.wait_all([
                yt.run_sort(self.out_f('dict') + 'dev_info', sort_by='key', sync=False),
                yt.run_sort(self.out_f('dict') + 'dev_info_yt', sort_by='devid', sync=False),
                yt.run_sort(self.out_f('dict') + 'uuid_dev_info', sort_by='key', sync=False),
                yt.run_sort(self.out_f('dict') + 'uuid_dev_info_yt', sort_by='uuid', sync=False),
                yt.run_sort(self.out_f('dict') + 'devid_hash', sort_by='devid', sync=False),
                yt.run_sort(self.out_f('dict') + 'devid_hash',
                            self.out_f('dict') + 'mmetric_devid', sort_by='mmetric_devid', sync=False),
                yt.run_sort(self.out_f('dict') + 'devid_mac', sort_by='mac', sync=False)
            ])

            mr.drop(self.out_f('mobile') + 'devid_mac_month_tmp')

            mr.set_generate_date(self.out_f('dict') + 'dev_info', self.date)
            mr.set_generate_date(self.out_f('dict') + 'dev_info_yt', self.date)
            mr.set_generate_date(self.out_f('dict') + 'uuid_dev_info', self.date)
            mr.set_generate_date(self.out_f('dict') + 'uuid_dev_info_yt', self.date)
            mr.set_generate_date(self.out_f('dict') + 'devid_hash', self.date)
            mr.set_generate_date(self.out_f('dict') + 'mmetric_devid', self.date)
            mr.set_generate_date(self.out_f('dict') + 'devid_mac', self.date)


    def output(self):
        return [yt_luigi.YtDateTarget(t, self.date) for t in
                [self.out_f('dict') + 'dev_info',
                 self.out_f('dict') + 'dev_info_yt',
                 self.out_f('dict') + 'uuid_dev_info',
                 self.out_f('dict') + 'devid_hash',
                 self.out_f('dict') + 'uuid_dev_info_yt',
                 self.out_f('dict') + 'mmetric_devid',
                 self.out_f('dict') + 'devid_mac']]


def prepare_uuid_info_for_upload(rec):
    yield to_yamr_format(rec[ids.UUID], rec)

def prepare_dev_info_for_upload(rec):
    yield to_yamr_format(rec[ids.CRYPTA_DEVICE_ID], rec)


class ConvertAppMetricaDayToOldFormats(yt_luigi.BaseYtTask):
    date = luigi.Parameter()

    def input_folders(self):
        return {
            'mobile': config.GRAPH_YT_OUTPUT_FOLDER + self.date + '/mobile/',
        }

    def output_folders(self):
        return {
            'mobile': config.GRAPH_YT_OUTPUT_FOLDER + self.date + '/mobile/',
        }

    def requires(self):
        return app_metrica_day.ImportAppMetrikaDayTask(date=self.date, run_date=self.date)

    def run(self):
        mr.mkdir(self.out_f('mobile'))

        utils.wait_all([
            yt.run_map(partial(to_dev_info_format, only_yamr_rec=True),
                       self.in_f('mobile') + 'dev_info_yt',
                       self.out_f('mobile') + 'dev_info',
                       sync=False),
            yt.run_map(partial(to_uuid_info_format, only_yamr_rec=True),
                       self.in_f('mobile') + 'uuid_info_yt',
                       self.out_f('mobile') + 'uuid_info',
                       sync=False),
        ])

        mr.sort_all([
            self.out_f('mobile') + 'dev_info',
            self.out_f('mobile') + 'uuid_info',
        ], sort_by=['key', 'subkey'])

        mr.set_generate_date(self.out_f('mobile') + 'dev_info', self.date)
        mr.set_generate_date(self.out_f('mobile') + 'uuid_info', self.date)


    def output(self):
        return [yt_luigi.YtDateTarget(self.out_f('mobile') + t, self.date)
                for t in ['dev_info', 'uuid_info']]


if __name__ == '__main__':
    yt.config.set_proxy(config.MR_SERVER)
    yt.config["tabular_data_format"] = yt.YsonFormat(process_table_index=True)

    import app_metrica_aggregate
    with yt.Transaction() as tr:
        mr.create_table_with_schema(
            '//home/crypta/production/state/graph/2017-08-19/mobile/dev_info_yt1',
            app_metrica_aggregate.dev_info_info_schema,
            tr,
            strict=False,
            recreate_if_exists=True,
        )
        #
        mr.create_table_with_schema(
            '//home/crypta/production/state/graph/2017-08-19/mobile/uuid_info_yt1',
            app_metrica_aggregate.uuid_info_schema,
            tr,
            strict=False,
            recreate_if_exists=True,
        )

    yt.run_merge('//home/crypta/production/state/graph/2017-08-19/mobile/dev_info_yt',
                 '//home/crypta/production/state/graph/2017-08-19/mobile/dev_info_yt1',
                 mode='sorted')

    yt.run_merge('//home/crypta/production/state/graph/2017-08-19/mobile/uuid_info_yt',
                 '//home/crypta/production/state/graph/2017-08-19/mobile/uuid_info_yt1',
                 mode='sorted')

    task = ConvertAppMetricaDayToOldFormats('2017-08-19')
    #
    luigi.build([task], workers=3, scheduler_port=8083)
