#!/usr/bin/python
# coding=utf-8

import os
import datetime

import luigi
import yt.wrapper as yt
from lib.luigi import yt_luigi
from rtcconf import config

from utils import utils
from utils import mr_utils as mr
from utils.yql_utils import run_yql
from data_imports.import_logs.app_metrica_day import ImportAppMetrikaDayTask

from v2.soup import soup_config
from v2.soup.soup_tables import SoupDumpTable


class Fuzzy2DayTask(yt_luigi.BaseYtTask):

    date = luigi.Parameter()
    run_date = luigi.Parameter()

    def __init__(self, *args, **kwargs):
        super(Fuzzy2DayTask, self).__init__(*args, **kwargs)

    def output_folders(self):
        return {
            'fuzzy': config.INDEVICE_YT_FOLDER + self.date + '/fuzzy/',
        }

    def requires(self):
        return [
            ImportAppMetrikaDayTask(date=self.date, run_date=self.run_date),
            yt_luigi.ExternalInput(config.LOG_FOLDERS['bs_watch'] + self.date), ]

    def run(self):
        run_yql('Fuzzy2Parser', dict(is_day_task='True', date_start=self.date, date_end=self.date), {
            'GRAPH_YT_OUTPUT_FOLDER': config.YT_OUTPUT_FOLDER,
            'INDEVICE_YT_FOLDER': config.INDEVICE_YT_FOLDER,
            'BSWATCH_LOG_DIR': config.LOG_FOLDERS['bs_watch'],
        })

        mr.drop(self.out_f('fuzzy') + 'fuzzy2_metrica')
        mr.drop(self.out_f('fuzzy') + 'fuzzy2_bswatch')

    def output(self):
        return [yt_luigi.YtTarget(t) for t in [self.out_f('fuzzy') + 'dev_yuid_fuzzy2']]


class Fuzzy2MonthTask(yt_luigi.BaseYtTask):

    date = luigi.Parameter()
    resources = {'fuzzy2_dump_lock': 1}

    def __init__(self, *args, **kwargs):
        super(Fuzzy2MonthTask, self).__init__(*args, **kwargs)
        self.soup_storages = [
            SoupDumpTable(soup_config.idfa_yuid_fuzzy2_indev, self.date),
            SoupDumpTable(soup_config.gaid_yuid_fuzzy2_indev, self.date)
        ]

    def requires(self):
        return [
            Fuzzy2DayTask(date=day, run_date=self.date)
            for day in utils.get_dates_before(self.date, int(config.STORE_DAYS))]

    def run(self):
        # run fuzzy2 as suprocess
        date_start = (
            datetime.datetime.strptime(self.date, '%Y-%m-%d') - datetime.timedelta(days=int(config.STORE_DAYS))
        ).strftime('%Y-%m-%d')

        run_yql('Fuzzy2Parser', dict(is_day_task='False', date_start=date_start, date_end=self.date), {
            'GRAPH_YT_OUTPUT_FOLDER': config.YT_OUTPUT_FOLDER,
            'INDEVICE_YT_FOLDER': config.INDEVICE_YT_FOLDER,
            'BSWATCH_LOG_DIR': config.LOG_FOLDERS['bs_watch'],
        })

        with yt.Transaction() as tr:
            for soup in self.soup_storages:
                soup.finalize(tr)

    def output(self):
        return [soup.as_target() for soup in self.soup_storages]
