from rtcconf import config


def get_device_matching_type_multi_source(rec, source_types):
    """
    Get two types of match for multi-source rec:
    - match type based on edge origin
    - match based on ua profile

    Method threats two types of matches as separate concepts.
    :return: tuple of match types
    """

    def origin_match_type(source_type):
        possible_match_types = config.DEVID_PAIR_TYPES_PERFECT_DICT[source_type].match_types
        if config.CROSSDEVICE in possible_match_types:
            # if cross-device is possible for source type, threat is as cross-device by origin
            return config.CROSSDEVICE
        elif config.INDEVICE in possible_match_types:
            # otherwise source type is in-device origin
            return config.INDEVICE
        else:
            raise Exception("Source might be at least in-device: %s" % source_type)

    ua_profile_match = ua_profile_match_type(rec)

    # if at least one edge is in-device, the whole rec is in-device
    for st in source_types:
        match_type = origin_match_type(st)
        if match_type == config.INDEVICE:
            return config.INDEVICE, ua_profile_match

    return config.CROSSDEVICE, ua_profile_match


def get_device_matching_type_single_source(rec, source_type):
    """
    Infers single match type decision based on two types of matches:
    - match type based on edge origin
    - match based on ua profile

    Methos seems mixing concepts but kept as is for compatibility
    :return: mixed match type
    """
    if source_type not in config.DEVID_PAIR_TYPES_PERFECT_DICT:
        raise ValueError('Source %s is no supported' % source_type)

    ua_profile_match = ua_profile_match_type(rec)

    possible_match_types = config.DEVID_PAIR_TYPES_PERFECT_DICT[source_type].match_types

    if ua_profile_match in [config.NOT_FOUND_DEV_INFO, config.NOT_FOUND_YUID_UA]:
        return ua_profile_match

    if config.CROSSDEVICE in possible_match_types:
        # may be either cross-device or in-device depending on user agent match
        if ua_profile_match == config.UA_PROFILE_MATCHES:
            return config.INDEVICE
        else:
            return config.CROSSDEVICE

    elif config.INDEVICE in possible_match_types:
        if ua_profile_match == config.UA_PROFILE_MATCHES:
            return config.INDEVICE
        else:
            # some error: indevice only is allowed by user agents don't match:
            # this is on
            return config.UA_PROFILE_NOT_MATCHES
    else:
        raise Exception("Source might be at least cross-device: %s" % source_type)


def ua_profile_match_type(rec):
    uuid_ua_profile = rec.get('uuid_ua_profile')
    yuid_ua_profile = rec.get('yuid_ua_profile')

    if not uuid_ua_profile:
        return config.NOT_FOUND_DEV_INFO

    elif not yuid_ua_profile:
        return config.NOT_FOUND_YUID_UA

    elif device_ua_profile_matches(yuid_ua_profile, uuid_ua_profile):
        return config.UA_PROFILE_MATCHES

    else:
        return config.UA_PROFILE_NOT_MATCHES


def device_ua_profile_matches(yuid_ua_profile, uuid_ua_profile):
    """
    checks, this uuid and yuid is in the same device

    How this works? Okay, everything is simple:
    each uuid_ua_profile and yuid_ua_profile looks like this - 'm|tablet|samsung|android|4.2.2'
    same device - when some parts of this strings are equeal. This parts:
       'm' (mobile) (must be m, because this is mobile) ;
       'tablet' (or smartphone, etc.) (type of device);
       'samsung' ('apple','huawei', etc.) (vendor);
       'adnroid' ('ios', etc.) (type of OS).
    '4.2.2' - this is the version of OS, it can be not equal.

    SO WHAT WE DO:
      - we getting this strings and split them
      - we check that there are both strings starts with 'm' (mobile)
      - we comform that there is each one of input splitted on five parts
      - and now we check all of 4 first parts are equal
    Last step: if we have not returned for this moment -> indevice, return True
              if not, that is not indevice (maybe cross device)

    CRYPTAIS-698 for perfect matching you can use os, os_version matching

    WARNING: DO NOT EDIT THIS WHILE YOU WILL ADD TESTING DATA!

    """
    # noinspection PyBroadException
    try:
        equal_frags = 4

        yuid_parts = yuid_ua_profile.split('|')
        uuid_parts = uuid_ua_profile.split('|')

        if not ('m' == yuid_parts[0] and 'm' == uuid_parts[0]):
            return False
        if not (len(yuid_parts) == 5 and len(uuid_parts) == 5):
            return False
        if not all(
                        yuid_parts[part_num] == uuid_parts[part_num]
                        for part_num in range(equal_frags)
        ):
            return False
        return True
    except:
        # throw message to log here
        return False


def pair_rec(devid, yuid, source_type, match_type, yuid_sources, table_index):
    return {'key': '_'.join((devid, yuid)), 'pair_type': 'd_y',
            'source_type': source_type, 'match_type': match_type,
            'pair_source': source_type + '_' + match_type,
            'yuid_sources': list(yuid_sources),
            '@table_index': table_index}
