from collections import defaultdict

import luigi
import yt.wrapper as yt

from data_imports.import_logs import device_yuids_mobreport
from data_imports.import_logs import device_yuids_oauth
from data_imports.import_logs import device_yuids_redir
from data_imports.import_logs import device_yuids_sdk
from data_imports.import_logs import device_yuids_tracking
from data_imports.import_logs import graph_access
from data_imports.import_logs import graph_barnavig
from data_imports.import_logs import graph_postclicks
from data_imports.import_logs import graph_sbapi_lookup
from data_imports.import_logs.watch_log import graph_watch_log
from lib.luigi import yt_luigi
from matching.device_matching.perfect.config import merge_configs_daily
from rtcconf import config
from utils import mr_utils as mr
from utils import utils


def merge_day_events_month(key, recs):
    dates_hits = defaultdict(int)
    last_ts = None

    for r in recs:
        dates_hits[r['dt']] += r['hits_count']

        rec_ts = r.get('last_ts')  # comes in reverse order, thus we store first
        if not last_ts and rec_ts:
            last_ts = rec_ts

    out_rec = dict(key)
    out_rec['last_ts'] = last_ts
    out_rec['dates'] = utils.default_to_regular(dates_hits)
    yield out_rec


def run_merge_all_dates(device_base_f, workdir, date):
    mr.mkdir(workdir + 'devid_raw_month')

    for m in merge_configs_daily:
        m.day_tables = mr.get_existing_date_tables(device_base_f, 'perfect/devid_raw_day/' + m.table_name(),
                                                   int(config.STORE_DAYS), before_date=date)

    # produce separate month table for every pair type
    utils.wait_all([
        yt.run_map_reduce(
            None,
            merge_day_events_month,
            m.day_tables,
            workdir + 'devid_raw_month/' + m.table_name(),
            reduce_by=[m.id1, m.id2, 'source_type'],
            sort_by=[m.id1, m.id2, 'source_type', 'last_ts'],
            sync=False
        ) for m in merge_configs_daily
    ])


class DeviceYuidsMergeMonthTask(yt_luigi.BaseYtTask):
    date = luigi.Parameter()

    def input_folders(self):
        return {
            'indevice': config.INDEVICE_YT_FOLDER,
        }

    def output_folders(self):
        return {
            'perfect': config.INDEVICE_YT_FOLDER + self.date + '/perfect/',
        }

    def requires(self):
        day_tasks = []
        for dt in utils.get_dates_before(self.date, int(config.STORE_DAYS)):
            day_tasks.append(device_yuids_redir.ImportRedirLogsDayTask(date=dt, run_date=self.date))
            day_tasks.append(
                device_yuids_oauth.ImportOauthLogsDayTask(date=dt, dict_date=self.date, run_date=self.date))
            day_tasks.append(graph_access.ImportAccessLogsDayTask(date=dt, run_date=self.date))
            day_tasks.append(graph_barnavig.ImportBarNavigDayTask(date=dt, run_date=self.date))
            day_tasks.append(graph_sbapi_lookup.ImportSbApiAccessLogDayTask(date=dt, run_date=self.date))
            day_tasks.append(graph_watch_log.ImportWatchLogDayTask(date=dt, run_date=self.date))
            day_tasks.append(device_yuids_tracking.ImportMobileTrackingLogsDayTask(date=dt, run_date=self.date))
            day_tasks.append(device_yuids_sdk.ImportSdkLogsDayTask(date=dt, run_date=self.date))
            day_tasks.append(device_yuids_mobreport.ImportMobReportLogsDayTask(date=dt, run_date=self.date))
            day_tasks.append(graph_postclicks.ImportPostclicksDailyTask(date=dt, run_date=self.date))

        return day_tasks

    def run(self):
        run_merge_all_dates(self.in_f('indevice'), self.out_f('perfect'), self.date)

    def output(self):
        return [yt_luigi.YtTarget(self.out_f('perfect') + 'devid_raw_month/' + m.table_name()) for m in merge_configs_daily]
