#!/usr/bin/python
# coding=utf-8

import datetime
from functools import partial

import luigi
import requests
import yt.wrapper as yt
from requests.packages.urllib3 import Retry

from lib.luigi import base_luigi_task
from lib.luigi import yt_luigi
from matching.human_matching.upload.graph_upload_prepare_vertices import add_experiment_id, \
    merge_bb_experiments, merge_bb_rm_cid, PrepareVerticesForUpload
from rtcconf import config
from utils import mr_utils as mr
from utils import utils


def upload_bb(table, name, delete_flag=False):
    yt_to_bb_url = config.YT_TO_BB_URL + "/tasks"
    params = {
        "yt_table": table,
        "bb_format": "tskv",
        "transport": "logbroker",
        "name": name,
        "delete": delete_flag,
        "dry_run": hasattr(config, 'BB_UPLOAD_DRY_RUN') and config.BB_UPLOAD_DRY_RUN
    }

    monrun_file_name = config.MONRUN_DATE_FOLDER + name + '_bb_upload.result'
    try:
        response = requests.post(yt_to_bb_url, data=params).json()
        task_id = response["task_id"]

        retries = Retry(backoff_factor=1.0)
        s = requests.Session()
        s.mount("http://", requests.adapters.HTTPAdapter(max_retries=retries))

        start_time = datetime.datetime.now()
        # try to receive terminal status for 6 hours
        while datetime.datetime.now() < start_time + datetime.timedelta(hours=8):
            task_state = s.get(yt_to_bb_url + "/" + str(task_id)).json()
            if task_state["status"] in ("ERROR", "BADYT"):
                utils.write_monrun(monrun_file_name, '2;%s upload failed: %s\n' % (name, task_state))
                break
            elif task_state["status"] == "DONE":
                utils.write_monrun(monrun_file_name, '0;OK\n')
                break
        else:  # no terminal status for 6 hours:
            utils.write_monrun(monrun_file_name, '2;%s upload failed: "6 hours expired, no final result"\n' % name)
    except Exception as e:
        utils.write_monrun(monrun_file_name, '2;%s upload failed: %s\n' % (name, str(e)))
        raise


class PrepareBBUpload(base_luigi_task.BaseTask):
    date = luigi.Parameter()
    vertices_configs = luigi.Parameter()

    def requires(self):
        return [PrepareVerticesForUpload(vertices_config=vc) for vc in self.vertices_configs]

    def merge_bb_experiments(self, upload_bb_folder):
        mr.mkdir(upload_bb_folder + 'experiments')

        experiment_tables = []
        ops = []
        # keep it in separate tables for later easy debug
        for vc in self.vertices_configs:
            experiment_id = config.VERTICES_EXPERIMENTS[vc.vertices_type]
            experiment_table = upload_bb_folder + 'experiments/cid_to_bb_' + vc.vertices_type + '_' + str(experiment_id)
            ops.append(yt.run_map(partial(add_experiment_id, experiment_id=experiment_id),
                                  # vc.get_upload_table(),  # upload diff
                                  vc.get_vertices_table(),  # upload all
                                  experiment_table, sync=False))

            experiment_tables.append(experiment_table)

        utils.wait_all(ops)

        upload_table = upload_bb_folder + 'cid_to_bb_with_experiments'

        yt.run_map_reduce(None, merge_bb_experiments,
                          experiment_tables,
                          [upload_table, upload_bb_folder + 'experiment_intersect_err'],
                          reduce_by=['crypta_id', 'key'])

        # TODO: disabled temporary to make testing work at weekend
        # if yt.row_count(upload_bb_folder + 'experiment_intersect_err') > 0:
        #     raise Exception('Crypta ids must not intersect among experiments')

        return upload_table

    def merge_bb_rm_cid(self, upload_bb_folder):
        rm_cid_tables = [vc.get_upload_rm_table() for vc in self.vertices_configs]
        mr.sort_all(rm_cid_tables, sort_by=['crypta_id', 'key'])
        upload_table = upload_bb_folder + 'rm_cid_to_bb'
        yt.run_reduce(partial(merge_bb_rm_cid), rm_cid_tables, upload_table, reduce_by=['crypta_id', 'key'])
        return upload_table

    def run(self):
        upload_bb_folder = config.YT_OUTPUT_FOLDER + self.date + '/upload_bb/'
        mr.mkdir(upload_bb_folder)

        self.merge_bb_experiments(upload_bb_folder)
        self.merge_bb_rm_cid(upload_bb_folder)

    def output(self):
        upload_bb_folder = config.YT_OUTPUT_FOLDER + self.date + '/upload_bb/'
        return [yt_luigi.YtTarget(upload_bb_folder + 'cid_to_bb_with_experiments'),
                yt_luigi.YtTarget(upload_bb_folder + 'rm_cid_to_bb', allow_empty=True)]


class UploadVerticesBBTask(base_luigi_task.BaseTask):
    date = luigi.Parameter()
    vertices_configs = luigi.Parameter()

    def requires(self):
        return PrepareBBUpload(self.date, self.vertices_configs)

    def run(self):
        upload_bb_folder = config.YT_OUTPUT_FOLDER + self.date + '/upload_bb/'

        if config.CRYPTA_UPLOAD_ENABLED == 'yes':
            upload_bb(upload_bb_folder + 'rm_cid_to_bb', 'rm_cid_to_bb_is', delete_flag=True)
            upload_bb(upload_bb_folder + 'cid_to_bb_with_experiments', 'cid_to_bb_is')

        yt_luigi.TodayFileTarget.done(config.MOBILE_TMP_FOLDER + 'upload_cid_bb_task', self.date)

    def output(self):
        return yt_luigi.TodayFileTarget(config.MOBILE_TMP_FOLDER + 'upload_cid_bb_task', self.date)


if __name__ == '__main__':
    yt.config.set_proxy(config.MR_SERVER)
    yt.config["tabular_data_format"] = yt.YsonFormat(process_table_index=True)

