import datetime

import luigi
import requests
import yt.wrapper as yt
from requests.packages.urllib3 import Retry

from lib.luigi import base_luigi_task
from lib.luigi import yt_luigi
from matching.device_matching.app_metrica import app_metrica_to_old_formats
from matching.device_matching.device_yuid_mix_perfect_fuzzy import DevidYuidMixPerfectFuzzy
from matching.human_matching.upload import graph_upload_prepare_vertices
from rtcconf import config
from utils import mr_utils as mr
from utils import utils


def map_vertices_to_is(rec):
    crypta_id = rec['crypta_id']
    id_key = rec['id']
    if rec['diff']:  # only diff goes to IS doe to performance reasons
        if rec['id_type'].startswith('yuid'):
            value = 's=' + rec['sources'] + '\tc=' + crypta_id + '\tt=' + rec['cid_type']
            yield {'key': id_key, 'subkey': '', 'value': value, '@table_index': 0}
        else:
            yield {'key': id_key, 'crypta_id': crypta_id, '@table_index': 1}


def map_indev_devid_yuid(rec):
    if rec.get('match_type') == config.INDEVICE:
        devid = rec['id1']
        yuid = rec['id2']
        yield {
            'key': devid,
            'subkey': yuid
        }


def join_dev_matching(devid_key, recs):
    main_yuid_recs, dev_cid_recs = mr.split_left_right(recs)

    values = []
    if dev_cid_recs:
        values.append('cid=%s' % dev_cid_recs[0]['crypta_id'])

    if main_yuid_recs:
        values.append('main_yuid=%s' % main_yuid_recs[0]['subkey'])

    yield {'key': devid_key['key'], 'subkey': '', 'value': '\t'.join(values)}


def upload_is(table, name):
    yt_to_bb_url = config.YT_TO_BB_URL + "/tasks"
    thread_count = config.IS_UPLOAD_THREAD_COUNTS[name]
    rps = config.IS_UPLOAD_RPS_LIMITS[name]
    params = {
        "yt_table": table,
        "bb_format": "tskv",
        "transport": "is",
        "threads": str(thread_count),
        "rps": str(rps),
        "type": name,
        "dry_run": hasattr(config, 'IS_UPLOAD_DRY_RUN') and config.IS_UPLOAD_DRY_RUN
    }

    monrun_file_name = config.MONRUN_DATE_FOLDER + name + '_is_upload.result'
    try:
        response = requests.post(yt_to_bb_url, data=params).json()
        task_id = response["task_id"]

        retries = Retry(backoff_factor=1.0)
        s = requests.Session()
        s.mount("http://", requests.adapters.HTTPAdapter(max_retries=retries))

        start_time = datetime.datetime.now()
        # try to receive terminal status for 6 hours
        while datetime.datetime.now() < start_time + datetime.timedelta(hours=8):
            task_state = s.get(yt_to_bb_url + "/" + str(task_id)).json()
            task_status = task_state["status"]

            if task_status in ("ERROR", "BADYT"):
                utils.write_monrun(monrun_file_name, '2;%s upload failed: %s\n' % (name, task_state))
                raise Exception("Upload of %s is failed: %s" % (table, task_status))

            elif task_status == "DONE":
                utils.write_monrun(monrun_file_name, '0;OK\n')
                break

        else:  # no terminal status for 6 hours:
            utils.write_monrun(monrun_file_name, '2;%s upload failed: "8 hours expired, no final result"\n' % name)
    except Exception as e:
        utils.write_monrun(monrun_file_name, '2;%s upload failed: %s\n' % (name, str(e)))
        raise


class PrepareCryptaMatchingUploadIS(yt_luigi.BaseYtTask):
    """
    Prepares all matching-related(vertices, main yuid for dev) IS uploads
    """
    date = luigi.Parameter()
    vertices_config = luigi.Parameter()

    def input_folders(self):
        return {
            'indevice': config.INDEVICE_YT_FOLDER + self.date + '/',
        }

    def output_folders(self):
        return {
            'upload_is': config.YT_OUTPUT_FOLDER + self.date + '/upload_is/'
        }

    def requires(self):
        return [graph_upload_prepare_vertices.PrepareVerticesForUpload(self.vertices_config),
                DevidYuidMixPerfectFuzzy(self.date)]

    def run(self):
        upload_is_folder = self.out_f('upload_is')
        mr.mkdir(upload_is_folder)

        yt.run_map(map_vertices_to_is, self.vertices_config.get_upload_table(),
                   [upload_is_folder + 'cid_to_is',
                    upload_is_folder + 'dev_cid'])

        yt.run_map(map_indev_devid_yuid, self.vertices_config.get_edges_table(), upload_is_folder + 'indev_devid_yuid')

        mr.sort_all([
            upload_is_folder + 'cid_to_is',
            upload_is_folder + 'dev_cid',
            upload_is_folder + 'indev_devid_yuid'
        ], sort_by='key')

        yt.run_reduce(join_dev_matching,
                          [self.in_f('indevice') + 'dev_yuid_info_ua',
                           upload_is_folder + 'dev_cid'],
                          upload_is_folder + 'dev_yuid_cid',
                          reduce_by='key')

        yt.run_sort(upload_is_folder + 'dev_yuid_cid', sort_by='key')

    def output(self):
        upload_is_folder = self.out_f('upload_is')
        return [yt_luigi.YtTarget(upload_is_folder + 'cid_to_is'),
                yt_luigi.YtTarget(upload_is_folder + 'dev_yuid_cid'),
                yt_luigi.YtTarget(upload_is_folder + 'indev_devid_yuid')]


class UploadISTask(base_luigi_task.BaseTask):
    """
    Aggregating task for all IS uploads
    """
    date = luigi.Parameter()
    vertices_config = luigi.Parameter()

    def requires(self):
        upload_is_folder = config.YT_OUTPUT_FOLDER + self.date + '/upload_is/'
        graph_mobile_folder = config.YT_OUTPUT_FOLDER + self.date + '/mobile/'
        prepare_is_task = PrepareCryptaMatchingUploadIS(self.date, self.vertices_config)
        import_mobile_task = app_metrica_to_old_formats.ConvertAppMetricaDayToOldFormats(date=self.date)


        # each task depends on previous in cascade to make serial execution
        dev_info_upload = UploadTableToIS(date=self.date,
                                          upload_id='dev_info',
                                          table=graph_mobile_folder + 'dev_info',
                                          required_tasks=[import_mobile_task])

        uuid_devid_upload = UploadTableToIS(date=self.date,
                                            upload_id='uuid_dev',
                                            table=graph_mobile_folder + 'uuid_info',
                                            required_tasks=[import_mobile_task, dev_info_upload])

        cid_yuid_upload = UploadTableToIS(date=self.date,
                                          upload_id='cid_yuid',
                                          table=upload_is_folder + 'cid_to_is',
                                          required_tasks=[prepare_is_task, uuid_devid_upload])

        dev_yuid_cid_upload = UploadTableToIS(date=self.date,
                                              upload_id='dev_yuid_cid',
                                              table=upload_is_folder + 'dev_yuid_cid',
                                              required_tasks=[prepare_is_task, cid_yuid_upload])

        indev_devid_yuid_upload = UploadTableToIS(date=self.date,
                                                  upload_id='yuid_devid',
                                                  table=upload_is_folder + 'indev_devid_yuid',
                                                  required_tasks=[prepare_is_task])

        return [dev_info_upload, uuid_devid_upload, cid_yuid_upload, dev_yuid_cid_upload, indev_devid_yuid_upload]

    def run(self):
        yt_luigi.TodayFileTarget.done(config.MOBILE_TMP_FOLDER + 'upload_is_task', self.date)

    def output(self):
        return yt_luigi.TodayFileTarget(config.MOBILE_TMP_FOLDER + 'upload_is_task', self.date)


class UploadTableToIS(base_luigi_task.BaseTask):
    """
    Uploads single table to IS
    """
    date = luigi.Parameter()
    upload_id = luigi.Parameter()
    table = luigi.Parameter()
    required_tasks = luigi.Parameter()

    def requires(self):
        return self.required_tasks

    def run(self):
        if config.CRYPTA_UPLOAD_ENABLED == 'yes':
            upload_is(self.table, self.upload_id)

        yt_luigi.TodayFileTarget.done(config.MOBILE_TMP_FOLDER + self.upload_id, self.date)

    def output(self):
        return yt_luigi.TodayFileTarget(config.MOBILE_TMP_FOLDER + self.upload_id, self.date)


if __name__ == '__main__':
    yt.config.set_proxy(config.MR_SERVER)
    import graph_vertices
    import graph_clustering

    yt.run_sort('//home/crypta/testing/state/indevice/2016-09-12/dev_yuid_info_ua',
                sort_by='key')

    dt = '2016-09-12'
    exact_vertices_task = graph_vertices.GraphVerticesExact(dt,
                                                            vertices_type='exact',
                                                            yuid_pairs_folder='pairs/')
    x = graph_clustering.ClusterVertices(exact_vertices_task.vertices_config, local_clustering_enabled=True)

    luigi.build([UploadISTask(dt, x.vertices_config)], workers=5, scheduler_port=8083)
