'''Luigi Task to collect and send metrics
about indevice relying on radius login data:
'''
import os
import sys
import json
import luigi
from lib.luigi import base_luigi_task
import yt.wrapper as yt
from lib.luigi import yt_luigi
from lib import graphite_sender

from functools import partial
from rtcconf import config
from radius_metrics import PrepareYuidRadiusData


ID_TYPES = (config.ID_TYPE_EMAIL, config.ID_TYPE_PHONE)
REFERENCE_SAMPLE_BASES = {
    config.ID_TYPE_EMAIL:
    '//crypta/production/state/extras/reference-bases/audi-mvideo-emails',
    config.ID_TYPE_PHONE:
    '//crypta/production/state/extras/reference-bases/sberbank_phones_hash',
}


def to_graphite(stats_table, date):
    stats = [json.loads(stat_record) for stat_record in yt.read_table(stats_table, format='json', raw=True)]
    metric_name = 'rlogin_coverage'
    metrics = []
    for r_stats in stats:
        device_type = r_stats.pop('device_type')
        for stat_name, value in r_stats.iteritems():
            stat_path = '.'.join([device_type, stat_name])
            metrics.append((metric_name, stat_path, value))
    graphite_sender.to_graphite_sender(metrics, date)


def reduce_indevice_pairs(key, recs, device_types=('desk',)):
    from itertools import combinations
    ua_profile = key['ua_profile']
    yuids = [rec['yuid'] for rec in recs if rec['device_type'] in device_types]
    for yuid_a, yuid_b in combinations(yuids, 2):
        pair = min(yuid_a, yuid_b) + '_' + max(yuid_a, yuid_b)
        yield {'pair': pair,
               'ua_profile': ua_profile}


def reduce_alldesk_pairs(key, recs, device_types=('desk',)):
    from itertools import combinations
    yuids = [rec['yuid'] for rec in recs if rec['device_type'] in device_types]
    for yuid_a, yuid_b in combinations(yuids, 2):
        pair = min(yuid_a, yuid_b) + '_' + max(yuid_a, yuid_b)
        yield {'pair': pair,}



def filter_exact_pairs(rec, rlogin_pairs):
    ua_profile_a = rec.get('id1_ua', None)
    ua_profile_b = rec.get('id2_ua', None)
    yuid_a, yuid_b = rec.get('pair').split('_')
    pair = min(yuid_a, yuid_b) + '_' + max(yuid_a, yuid_b)
    if ua_profile_a == ua_profile_b and (pair, ua_profile_a) in rlogin_pairs:
        yuid_a, yuid_b = pair.split('_')
        yield {'pair': pair,
               'ua_profile': ua_profile_a}


def indevice_coverage(device_type, workdir, yuid_rlogin_pairs, exact_pairs):
    '''
    * % of in-device desktop rlogin pairs covered by exact pairs.
    '''
    stats = {'device_type': device_type}
    yt.run_sort(yuid_rlogin_pairs, sort_by=['login', 'ua_profile'])
    rlog_indev_pairs_table = os.path.join(workdir, 'tmp_rlogin_indev_pairs')
    yt.run_reduce(partial(reduce_indevice_pairs, device_types=(device_type,)),
                  yuid_rlogin_pairs,
                  rlog_indev_pairs_table,
                  reduce_by=['login', 'ua_profile'])
    rlogin_pairs = set((rec['pair'], rec['ua_profile']) for rec in
                       (json.loads(rec) for rec in
                        yt.read_table(rlog_indev_pairs_table, format='json', raw=True)))
    exact_filtered_pairs = os.path.join(workdir, 'tmp_exact_filtered_pairs')
    yt.run_map(partial(filter_exact_pairs, rlogin_pairs=rlogin_pairs),
               exact_pairs,
               exact_filtered_pairs)

    if len(rlogin_pairs) == 0:
        stats['indevice'] = 0
    else:
        stats['indevice'] = \
            100.0 * yt.row_count(exact_filtered_pairs) / len(rlogin_pairs)

    yt.remove(exact_filtered_pairs)

    return stats


def write_stats(table, stats):
    yt.write_table(table, [json.dumps(s) for s in stats], format='json', raw=True)


class RadiusMetricsIndevice(base_luigi_task.BaseTask):
    '''Luigi Task to measure stat indevice coverage.
    '''
    date = luigi.Parameter()
    vertices_config = luigi.Parameter()

    def requires(self):
        return [PrepareYuidRadiusData(self.date),
                self.vertices_config.producing_task]

    def run(self):
        pretty_pairs_table = self.vertices_config.get_edges_table()
        output_folder = os.path.join(config.RADIUS_METRICS_YT_FOLDER,
                                     self.date)
        rlogin_yuid_table = os.path.join(output_folder,
                                         'yuid_rlogin',
                                         'yuid_rlogin_final_stable')
        stats = []
        # for device_type in ...
        stats.append(indevice_coverage('desk',
                                       os.path.join(output_folder, 'yuid_rlogin'),
                                       rlogin_yuid_table,
                                       pretty_pairs_table))
        stats_table = os.path.join(output_folder, 'coverage_stats')
        write_stats(stats_table, stats)
        to_graphite(stats_table, self.date)

    def output(self):
        output_folder = os.path.join(config.RADIUS_METRICS_YT_FOLDER,
                                     self.date)
        stats_table = os.path.join(output_folder, 'coverage_stats')
        return [yt_luigi.YtTarget(stats_table),]


if __name__ == '__main__':
    import luigi
    yt.config.set_proxy(config.MR_SERVER)
    dt = sys.argv[1]

    luigi.run(['RadiusMetricsIndevice',
               '--date', dt,
               # '--name', 'rlogin_stats',
               '--workers', '10',
               '--local-scheduler'])
